<?php

namespace RubikaBot;



use PDO; 

use PDOException;



class MainDatabase {

	private PDO $pdo;



	public function __construct(?string $dbFilePath = null) {

		$dbFile = $dbFilePath ?? __DIR__ . '/../data/rubikabot_main.sqlite';

		$needInit = !file_exists($dbFile);

		

		// Ensure the directory exists

        $dbDir = dirname($dbFile);

        if (!is_dir($dbDir)) {

            mkdir($dbDir, 0777, true);

        }



		try {

			$this->pdo = new PDO('sqlite:' . $dbFile);

			$this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

		} catch (PDOException $e) {

			error_log('MainDB connection failed: ' . $e->getMessage());

			throw $e;

		}

		

		if ($needInit) {

			error_log("MainDatabase: Initializing new main database");

			$this->initSchema();

		}

		

	}



	public function getPdo(): PDO { return $this->pdo; }



	private function initSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				chat_id TEXT UNIQUE, 

				sender_id TEXT,

				balance REAL DEFAULT 0, 

				is_banned INTEGER DEFAULT 0, 

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP

			)',

			'CREATE TABLE IF NOT EXISTS bots (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				user_id INTEGER, 

				bot_token TEXT UNIQUE, 

				bot_username TEXT, 

				bot_directory TEXT, 

				bot_type TEXT DEFAULT "series", 

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

				FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE

			)',

			'CREATE TABLE IF NOT EXISTS transactions (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				user_id INTEGER, 

				amount REAL, 

				description TEXT, 

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

				FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE

			)',

			'CREATE INDEX IF NOT EXISTS idx_users_chat_id ON users(chat_id)',

			'CREATE INDEX IF NOT EXISTS idx_bots_token ON bots(bot_token)',

			'CREATE INDEX IF NOT EXISTS idx_transactions_user_id ON transactions(user_id)'

		];

		

		foreach ($schema as $sql) {

			try {

				$this->pdo->exec($sql);

				error_log('MainDB: Created table/index: ' . substr($sql, 0, 50) . '...');

			} catch (PDOException $e) {

				error_log('MainDB: Error creating table: ' . $e->getMessage());

			}

		}

	}



	private function runMigrations(): void {

		// Check if main tables exist and create them if they don't

		try {

			$tables = ['users', 'bots', 'transactions'];

			foreach ($tables as $table) {

				$stmt = $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='$table'");

				if (!$stmt->fetch()) {

					error_log("MainDB: Missing table '$table', recreating schema");

					$this->initSchema();

					break;

				}

			}

		} catch (PDOException $e) {

			error_log('MainDB: Error checking tables: ' . $e->getMessage());

		}



		// Migration 1: Add bot_type column if it doesn't exist

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(bots)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			if (!in_array('bot_type', $columnNames)) {

				$this->pdo->exec('ALTER TABLE bots ADD COLUMN bot_type TEXT DEFAULT "series"');

				error_log('MainDB: Added bot_type column to bots table');

			}

		} catch (PDOException $e) {

			error_log('MainDB: Migration error for bot_type column: ' . $e->getMessage());

		}



		// Migration 2: Ensure is_banned column exists in users table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(users)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			if (!in_array('is_banned', $columnNames)) {

				$this->pdo->exec('ALTER TABLE users ADD COLUMN is_banned INTEGER DEFAULT 0');

				error_log('MainDB: Added is_banned column to users table');

			}

		} catch (PDOException $e) {

			error_log('MainDB: Migration error for is_banned column: ' . $e->getMessage());

		}



		// Migration 3: Add status column to bots table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(bots)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			if (!in_array('status', $columnNames)) {

				$this->pdo->exec('ALTER TABLE bots ADD COLUMN status TEXT DEFAULT "active"');

				error_log('MainDB: Added status column to bots table');

			}

		} catch (PDOException $e) {

			error_log('MainDB: Migration error for status column: ' . $e->getMessage());

		}



		// Migration 4: Add sender_id column to users table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(users)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			if (!in_array('sender_id', $columnNames)) {

				$this->pdo->exec('ALTER TABLE users ADD COLUMN sender_id TEXT');

				error_log('MainDB: Added sender_id column to users table');

			}

		} catch (PDOException $e) {

			error_log('MainDB: Migration error for sender_id column: ' . $e->getMessage());

		}

	}



	// Helper methods for main bot functionality

	public function addUser(string $chatId, ?string $senderId = null): bool {

		$stmt = $this->pdo->prepare("INSERT OR IGNORE INTO users (chat_id, sender_id) VALUES (?, ?)");

		return $stmt->execute([$chatId, $senderId]);

	}



	public function getUser(string $chatId): ?array {

		$stmt = $this->pdo->prepare("SELECT * FROM users WHERE chat_id = ?");

		$stmt->execute([$chatId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function updateUserBalance(string $chatId, float $newBalance): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET balance = ? WHERE chat_id = ?");

		return $stmt->execute([$newBalance, $chatId]);

	}



	public function getUserById(int $userId): ?array {

		$stmt = $this->pdo->prepare("SELECT * FROM users WHERE id = ?");

		$stmt->execute([$userId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function banUser(int $userId): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET is_banned = 1 WHERE id = ?");

		return $stmt->execute([$userId]);

	}



	public function unbanUser(int $userId): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET is_banned = 0 WHERE id = ?");

		return $stmt->execute([$userId]);

	}



	public function isUserBanned(string $chatId): bool {

		$stmt = $this->pdo->prepare("SELECT is_banned FROM users WHERE chat_id = ?");

		$stmt->execute([$chatId]);

		$result = $stmt->fetch(PDO::FETCH_ASSOC);

		return $result ? (bool)($result['is_banned'] ?? false) : false;

	}



	public function createBot(int $userId, string $botToken, string $botUsername, string $botDirectory, string $botType): bool {

		$stmt = $this->pdo->prepare("INSERT INTO bots (user_id, bot_token, bot_username, bot_directory, bot_type) VALUES (?, ?, ?, ?, ?)");

		return $stmt->execute([$userId, $botToken, $botUsername, $botDirectory, $botType]);

	}



	public function getUserBots(int $userId): array {

		$stmt = $this->pdo->prepare("SELECT * FROM bots WHERE user_id = ? ORDER BY created_at DESC");

		$stmt->execute([$userId]);

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function getBotById(int $botId): ?array {

		$stmt = $this->pdo->prepare("SELECT * FROM bots WHERE id = ?");

		$stmt->execute([$botId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function updateBotConfig(int $botId, string $userSenderId, string $chatId): bool {

		// Get bot info

		$bot = $this->getBotById($botId);

		if (!$bot) {

			error_log("MainDB: Bot not found with ID: $botId");

			return false;

		}



		// Extract directory name from bot_directory path

		$botDirectoryName = basename($bot['bot_directory']);

		$configPath = __DIR__ . '/../created_bots/' . $botDirectoryName . '/config.php';

		

		if (!file_exists($configPath)) {

			error_log("MainDB: Config file not found: $configPath");

			return false;

		}



		// Update config file with user's sender_id

		$configContent = "<?php\n";

		$configContent .= "define('BOT_TOKEN', '".$bot['bot_token']."');\n";

		$configContent .= "define('ADMIN_SENDER_ID', '".$userSenderId."');\n";

		$configContent .= "define('ADMIN_CHAT_ID', '".$chatId."');\n";

		$configContent .= "define('DB_FILE_PATH', __DIR__ . '/".$botDirectoryName."_".$bot['bot_type'].".sqlite');\n";

		$configContent .= "define('BOT_TYPE', '".$bot['bot_type']."');\n\n";



		$result = file_put_contents($configPath, $configContent);

		if ($result === false) {

			error_log("MainDB: Failed to write config file: $configPath");

		} else {

			error_log("MainDB: Successfully updated config for bot $botDirectoryName");

		}

		

		return $result !== false;

	}



	public function updateBotToken(int $botId, string $userSenderId, string $chatId, string $newToken): bool {

		// Get bot info

		$bot = $this->getBotById($botId);

		if (!$bot) {

			error_log("MainDB: Bot not found with ID: $botId for token update");

			return false;

		}



		// Update token in database

		$stmt = $this->pdo->prepare("UPDATE bots SET bot_token = ? WHERE id = ?");

		$tokenUpdated = $stmt->execute([$newToken, $botId]);



		if (!$tokenUpdated) {

			error_log("MainDB: Failed to update token in database for bot ID: $botId");

			return false;

		}



		// Extract directory name from bot_directory path

		$botDirectoryName = basename($bot['bot_directory']);

		$configPath = __DIR__ . '/../created_bots/' . $botDirectoryName . '/config.php';

		

		if (!file_exists($configPath)) {

			error_log("MainDB: Config file not found for token update: $configPath");

			return false;

		}



		$configContent = "<?php\n";

		$configContent .= "define('BOT_TOKEN', '{$newToken}');\n";

		$configContent .= "define('ADMIN_SENDER_ID', '{$userSenderId}');\n";

		$configContent .= "define('ADMIN_CHAT_ID', '{$chatId}');\n";

		$configContent .= "define('DB_FILE_PATH', __DIR__ . '/{$botDirectoryName}_{$bot['bot_type']}.sqlite');\n";

		$configContent .= "define('BOT_TYPE', '{$bot['bot_type']}');\n\n";



		$result = file_put_contents($configPath, $configContent);

		if ($result === false) {

			error_log("MainDB: Failed to write config file for token update: $configPath");

		} else {

			error_log("MainDB: Successfully updated token and config for bot $botDirectoryName");

		}



		return $result !== false;

	}



	public function addTransaction(int $userId, float $amount, string $description): bool {

		$stmt = $this->pdo->prepare("INSERT INTO transactions (user_id, amount, description) VALUES (?, ?, ?)");

		return $stmt->execute([$userId, $amount, $description]);

	}



	public function getStats(): array {

		$stats = [];

		$stats['users'] = $this->pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

		$stats['banned_users'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE is_banned = 1")->fetchColumn();

		$stats['bots'] = $this->pdo->query("SELECT COUNT(*) FROM bots")->fetchColumn();

		$stats['active_bots'] = $this->pdo->query("SELECT COUNT(*) FROM bots WHERE status = 'active'")->fetchColumn();

		$stats['transactions'] = $this->pdo->query("SELECT COUNT(*) FROM transactions")->fetchColumn();

		return $stats;

	}

}

?>

