<?php



require_once __DIR__ . '/../src/RubikaBot.php';

require_once __DIR__ . '/../src/MainDatabase.php';



use RubikaBot\Bot;

use RubikaBot\MainDatabase;



/**

 * Mass Advertisement System for Created Bots

 * 

 * این فایل برای ارسال تبلیغات همگانی روی تمام ربات‌های ساخته شده طراحی شده است

 * هر 5 ثانیه برای 10 نفر پیام ارسال می‌کند و نتیجه را در رباتساز گزارش می‌دهد

 */



// --- ⚙️ Configuration ---

$MAIN_BOT_TOKEN = 'BABHF0QEVLLYPOXMTFUWGQPXRKAQFQHKPACQLLMFKRLPZGCRNJTPCVXXWYNJDCLY'; // توکن رباتساز اصلی

$ADMIN_CHAT_ID = 'b0HSmX80Cjo0406abec1627e9d44d855'; // CHAT_ID ادمین

$ADMIN_SENDER_ID = 'u0HSmX805a5516d2f2dc058ea3a97712'; // GUID ادمین



// --- 📡 Advertisement Configuration ---

define('AD_BATCH_SIZE', 10); // هر چند کاربر صبر کند

define('AD_DELAY_SECONDS', 5); // چند ثانیه صبر کند

define('AD_PROGRESS_INTERVAL', 50); // هر چند پیام گزارش پیشرفت بدهد

define('LOG_FILE', 'advertisement_log.txt'); // فایل لاگ



// --- 📂 Directories ---

define('BASE_DIR', __DIR__ . '/..');

define('CREATED_BOTS_DIR', BASE_DIR . '/created_bots');



class MassAdvertisement {

    private PDO $pdo;

    private Bot $mainBot;

    private string $adminChatId;

    private array $stats = [

        'total_bots' => 0,

        'total_users' => 0,

        'successful_sends' => 0,

        'failed_sends' => 0,

        'processed_bots' => 0,

        'start_time' => 0,

        'current_bot' => ''

    ];



    public function __construct() {

        global $MAIN_BOT_TOKEN, $ADMIN_CHAT_ID;

        

        // اتصال به دیتابیس اصلی

        $db = new MainDatabase();

        $this->pdo = $db->getPdo();

        

        // ربات اصلی برای گزارش‌گیری

        $this->mainBot = new Bot($MAIN_BOT_TOKEN);

        $this->adminChatId = $ADMIN_CHAT_ID;

        

        $this->stats['start_time'] = time();

        $this->log("Mass Advertisement System Started");

    }



    /**

     * ارسال تبلیغات همگانی

     */

    public function sendMassAdvertisement(string $message, ?string $forwardFromChatId = null, ?string $forwardMessageId = null): array {

        $this->log("Starting mass advertisement process");

        

        // دریافت لیست تمام ربات‌ها

        $bots = $this->getAllBots();

        $this->stats['total_bots'] = count($bots);

        

        if (empty($bots)) {

            $this->sendAdminNotification("❌ هیچ رباتی برای ارسال تبلیغات یافت نشد!");

            return $this->getStats();

        }



        $this->sendAdminNotification("🚀 شروع ارسال تبلیغات همگانی\n\n📊 تعداد ربات‌ها: {$this->stats['total_bots']}\n⏱️ تنظیمات: هر " . AD_BATCH_SIZE . " کاربر، " . AD_DELAY_SECONDS . " ثانیه صبر");



        // پردازش هر ربات

        foreach ($bots as $bot) {

            $this->processBotAdvertisement($bot, $message, $forwardFromChatId, $forwardMessageId);

            $this->stats['processed_bots']++;

            

            // گزارش پیشرفت

            if ($this->stats['processed_bots'] % 5 === 0) {

                $this->sendProgressReport();

            }

        }



        // ارسال گزارش نهایی

        $this->sendFinalReport();

        

        return $this->getStats();

    }



    /**

     * پردازش تبلیغات برای یک ربات خاص

     */

    private function processBotAdvertisement(array $botInfo, string $message, ?string $forwardFromChatId = null, ?string $forwardMessageId = null): void {

        $this->stats['current_bot'] = $botInfo['bot_username'];

        $this->log("Processing bot: {$botInfo['bot_username']}");



        try {

            // ایجاد اتصال به ربات

            $bot = new Bot($botInfo['bot_token']);

            

            // دریافت کاربران این ربات

            $users = $this->getBotUsers($botInfo['bot_directory']); // Pass bot_directory instead of bot_username

            

            if (empty($users)) {

                $this->log("No users found for bot: {$botInfo['bot_username']}");

                return;

            }



            $this->stats['total_users'] += count($users);

            $botSuccessCount = 0;

            $botFailCount = 0;



            // ارسال پیام به کاربران

            foreach ($users as $index => $userChatId) {

                try {

                    $success = false;

                    

                    if ($forwardFromChatId && $forwardMessageId) {

                        // فوروارد پیام

                        $result = $bot->forwardFrom($forwardFromChatId)

                                     ->messageId($forwardMessageId)

                                     ->forwardTo($userChatId)

                                     ->forward();

                        $success = $result && isset($result['status']) && $result['status'] === 'OK';

                    } else {

                        // ارسال پیام متنی

                        $result = $bot->chat($userChatId)->message($message)->send();

                        $success = $result && isset($result['status']) && $result['status'] === 'OK';

                    }



                    if ($success) {

                        $botSuccessCount++;

                        $this->stats['successful_sends']++;

                    } else {

                        $botFailCount++;

                        $this->stats['failed_sends']++;

                    }



                } catch (Exception $e) {

                    $botFailCount++;

                    $this->stats['failed_sends']++;

                    $this->log("Failed to send to user {$userChatId} in bot {$botInfo['bot_username']}: " . $e->getMessage());

                }



                // تأخیر بعد از هر گروه

                if (($index + 1) % AD_BATCH_SIZE === 0) {

                    sleep(AD_DELAY_SECONDS);

                }

            }



            $this->log("Bot {$botInfo['bot_username']} completed: Success={$botSuccessCount}, Failed={$botFailCount}");



        } catch (Exception $e) {

            $this->log("Error processing bot {$botInfo['bot_username']}: " . $e->getMessage());

        }

    }



    /**

     * دریافت لیست تمام ربات‌ها

     */

    private function getAllBots(): array {

        $stmt = $this->pdo->prepare("SELECT bot_token, bot_username, bot_directory FROM bots ORDER BY id ASC");

        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);

    }



    /**

     * دریافت کاربران یک ربات خاص

     */

    private function getBotUsers(string $botDirectory): array {

        // Use basename to get just the bot directory name from full path

        $botDirName = basename($botDirectory);

        $botDbFile = CREATED_BOTS_DIR . '/' . $botDirName . '/' . $botDirName . '_series.sqlite';

        

        if (!file_exists($botDbFile)) {

            $this->log("Database not found for bot directory: {$botDirectory} at {$botDbFile}");

            return [];

        }



        try {

            $botPdo = new PDO('sqlite:' . $botDbFile);

            $botPdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            

            $stmt = $botPdo->prepare("SELECT chat_id FROM users ORDER BY joined_at ASC");

            $stmt->execute();

            

            return $stmt->fetchAll(PDO::FETCH_COLUMN);

            

        } catch (PDOException $e) {

            $this->log("Error accessing database for bot directory {$botDirectory}: " . $e->getMessage());

            return [];

        }

    }



    /**

     * ارسال گزارش پیشرفت

     */

    private function sendProgressReport(): void {

        $elapsed = time() - $this->stats['start_time'];

        $progress = round(($this->stats['processed_bots'] / $this->stats['total_bots']) * 100, 1);

        

        $message = "📊 گزارش پیشرفت تبلیغات\n\n";

        $message .= "🤖 ربات‌های پردازش شده: {$this->stats['processed_bots']}/{$this->stats['total_bots']} ({$progress}%)\n";

        $message .= "👥 کل کاربران: {$this->stats['total_users']}\n";

        $message .= "✅ ارسال موفق: {$this->stats['successful_sends']}\n";

        $message .= "❌ ارسال ناموفق: {$this->stats['failed_sends']}\n";

        $message .= "⏱️ زمان سپری شده: {$elapsed} ثانیه\n";

        $message .= "🔄 ربات فعلی: {$this->stats['current_bot']}";



        $this->sendAdminNotification($message);

    }



    /**

     * ارسال گزارش نهایی

     */

    private function sendFinalReport(): void {

        $totalTime = time() - $this->stats['start_time'];

        $successRate = $this->stats['total_users'] > 0 

            ? round(($this->stats['successful_sends'] / ($this->stats['successful_sends'] + $this->stats['failed_sends'])) * 100, 1) 

            : 0;



        $message = "✅ تبلیغات همگانی کامل شد!\n\n";

        $message .= "📊 گزارش نهایی:\n";

        $message .= "🤖 تعداد ربات‌ها: {$this->stats['total_bots']}\n";

        $message .= "👥 کل کاربران: {$this->stats['total_users']}\n";

        $message .= "✅ ارسال موفق: {$this->stats['successful_sends']}\n";

        $message .= "❌ ارسال ناموفق: {$this->stats['failed_sends']}\n";

        $message .= "📈 نرخ موفقیت: {$successRate}%\n";

        $message .= "⏱️ زمان کل: {$totalTime} ثانیه\n\n";

        $message .= "🎉 عملیات با موفقیت انجام شد!";



        $this->sendAdminNotification($message);

        $this->log("Mass advertisement completed. Stats: " . json_encode($this->stats));

    }



    /**

     * ارسال اطلاعیه به ادمین

     */

    private function sendAdminNotification(string $message): void {

        try {

            $this->mainBot->chat($this->adminChatId)->message($message)->send();

        } catch (Exception $e) {

            $this->log("Failed to send admin notification: " . $e->getMessage());

        }

    }



    /**

     * ثبت لاگ

     */

    private function log(string $message): void {

        $timestamp = date('Y-m-d H:i:s');

        $logMessage = "[{$timestamp}] {$message}\n";

        file_put_contents(LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);

        error_log("MassAd: " . $message);

    }



    /**

     * دریافت آمار

     */

    public function getStats(): array {

        return $this->stats;

    }

}



// --- 🎯 CLI Usage ---

if (php_sapi_name() === 'cli') {

    echo "🚀 Mass Advertisement System\n";

    echo "===========================\n\n";

    

    if ($argc < 2) {

        echo "Usage: php mass_advertisement.php \"Your advertisement message\"\n";

        echo "   or: php mass_advertisement.php --forward chat_id message_id\n\n";

        echo "Examples:\n";

        echo "  php mass_advertisement.php \"🎉 سلام! این یک تبلیغ تستی است\"\n";

        echo "  php mass_advertisement.php --forward u0ABC123 c0DEF456\n";

        exit(1);

    }

    

    $massAd = new MassAdvertisement();

    

    if ($argv[1] === '--forward' && $argc >= 4) {

        // حالت فوروارد

        $fromChatId = $argv[2];

        $messageId = $argv[3];

        echo "Starting mass forward from {$fromChatId}, message {$messageId}...\n";

        $stats = $massAd->sendMassAdvertisement("", $fromChatId, $messageId);

    } else {

        // حالت پیام متنی

        $message = $argv[1];

        echo "Starting mass advertisement with message: {$message}\n";

        $stats = $massAd->sendMassAdvertisement($message);

    }

    

    echo "\n✅ Completed!\n";

    echo "📊 Final Stats:\n";

    echo "   Total Bots: {$stats['total_bots']}\n";

    echo "   Total Users: {$stats['total_users']}\n";

    echo "   Successful: {$stats['successful_sends']}\n";

    echo "   Failed: {$stats['failed_sends']}\n";

    echo "   Duration: " . (time() - $stats['start_time']) . " seconds\n";

}



// --- 🌐 Web Interface ---

if (isset($_POST['action']) && $_POST['action'] === 'send_advertisement') {

    header('Content-Type: application/json');

    

    // بررسی مجوز ادمین

    $senderId = $_POST['sender_id'] ?? '';

    if ($senderId !== $ADMIN_SENDER_ID) {

        echo json_encode(['success' => false, 'message' => 'Unauthorized']);

        exit;

    }

    

    $message = $_POST['message'] ?? '';

    $forwardFromChatId = $_POST['forward_from_chat_id'] ?? null;

    $forwardMessageId = $_POST['forward_message_id'] ?? null;

    

    if (empty($message) && (empty($forwardFromChatId) || empty($forwardMessageId))) {

        echo json_encode(['success' => false, 'message' => 'Message or forward parameters required']);

        exit;

    }

    

    try {

        $massAd = new MassAdvertisement();

        $stats = $massAd->sendMassAdvertisement($message, $forwardFromChatId, $forwardMessageId);

        echo json_encode(['success' => true, 'stats' => $stats]);

    } catch (Exception $e) {

        echo json_encode(['success' => false, 'message' => $e->getMessage()]);

    }

    

    exit;

}



?>



<!DOCTYPE html>

<html lang="fa" dir="rtl">

<head>

    <meta charset="UTF-8">

    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <title>سیستم تبلیغات همگانی</title>

    <style>

        body {

            font-family: Tahoma, Arial, sans-serif;

            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);

            margin: 0;

            padding: 20px;

            color: #333;

        }

        .container {

            max-width: 800px;

            margin: 0 auto;

            background: white;

            border-radius: 15px;

            padding: 30px;

            box-shadow: 0 10px 30px rgba(0,0,0,0.2);

        }

        .header {

            text-align: center;

            margin-bottom: 30px;

            border-bottom: 2px solid #f0f0f0;

            padding-bottom: 20px;

        }

        .form-group {

            margin-bottom: 20px;

        }

        label {

            display: block;

            margin-bottom: 5px;

            font-weight: bold;

            color: #555;

        }

        textarea, input {

            width: 100%;

            padding: 12px;

            border: 2px solid #ddd;

            border-radius: 8px;

            font-size: 14px;

            font-family: inherit;

            box-sizing: border-box;

        }

        textarea {

            min-height: 100px;

            resize: vertical;

        }

        .btn {

            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);

            color: white;

            padding: 12px 30px;

            border: none;

            border-radius: 8px;

            cursor: pointer;

            font-size: 16px;

            font-weight: bold;

            transition: transform 0.2s;

        }

        .btn:hover {

            transform: translateY(-2px);

        }

        .btn:disabled {

            background: #ccc;

            cursor: not-allowed;

            transform: none;

        }

        .tabs {

            display: flex;

            margin-bottom: 20px;

            border-bottom: 1px solid #ddd;

        }

        .tab {

            padding: 10px 20px;

            cursor: pointer;

            border-bottom: 2px solid transparent;

            transition: all 0.3s;

        }

        .tab.active {

            border-bottom-color: #667eea;

            color: #667eea;

            font-weight: bold;

        }

        .tab-content {

            display: none;

        }

        .tab-content.active {

            display: block;

        }

        .result {

            margin-top: 20px;

            padding: 15px;

            border-radius: 8px;

            display: none;

        }

        .success {

            background-color: #d4edda;

            border: 1px solid #c3e6cb;

            color: #155724;

        }

        .error {

            background-color: #f8d7da;

            border: 1px solid #f5c6cb;

            color: #721c24;

        }

        .stats {

            display: grid;

            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));

            gap: 15px;

            margin-top: 15px;

        }

        .stat-card {

            background: #f8f9fa;

            padding: 15px;

            border-radius: 8px;

            text-align: center;

            border: 1px solid #e9ecef;

        }

        .stat-number {

            font-size: 24px;

            font-weight: bold;

            color: #667eea;

        }

        .stat-label {

            font-size: 12px;

            color: #666;

            margin-top: 5px;

        }

    </style>

</head>

<body>

    <div class="container">

        <div class="header">

            <h1>🚀 سیستم تبلیغات همگانی</h1>

            <p>ارسال تبلیغات به تمام کاربران ربات‌های ساخته شده</p>

        </div>



        <div class="tabs">

            <div class="tab active" onclick="switchTab('message')">📝 پیام متنی</div>

            <div class="tab" onclick="switchTab('forward')">📤 فوروارد پیام</div>

        </div>



        <form id="advertisementForm">

            <div id="message-tab" class="tab-content active">

                <div class="form-group">

                    <label for="advertisement_message">🔤 متن تبلیغات:</label>

                    <textarea id="advertisement_message" name="message" placeholder="متن تبلیغات خود را اینجا بنویسید...

مثال:

🎉 پیشنهاد ویژه!

50% تخفیف روی تمام محصولات

⏰ فقط تا پایان هفته"></textarea>

                </div>

            </div>



            <div id="forward-tab" class="tab-content">

                <div class="form-group">

                    <label for="forward_from_chat_id">🆔 شناسه چت مبدأ:</label>

                    <input type="text" id="forward_from_chat_id" name="forward_from_chat_id" placeholder="u0ABC123... یا c0DEF456...">

                </div>

                <div class="form-group">

                    <label for="forward_message_id">📨 شناسه پیام:</label>

                    <input type="text" id="forward_message_id" name="forward_message_id" placeholder="شناسه پیام برای فوروارد">

                </div>

            </div>



            <div class="form-group">

                <button type="submit" class="btn" id="sendBtn">🚀 شروع ارسال تبلیغات</button>

            </div>

        </form>



        <div id="result" class="result">

            <div id="resultMessage"></div>

            <div id="stats" class="stats"></div>

        </div>

    </div>



    <script>

        let currentTab = 'message';



        function switchTab(tab) {

            // Update tab buttons

            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));

            document.querySelector(`.tab:nth-child(${tab === 'message' ? '1' : '2'})`).classList.add('active');

            

            // Update tab contents

            document.querySelectorAll('.tab-content').forEach(t => t.classList.remove('active'));

            document.getElementById(`${tab}-tab`).classList.add('active');

            

            currentTab = tab;

        }



        document.getElementById('advertisementForm').addEventListener('submit', async function(e) {

            e.preventDefault();

            

            const sendBtn = document.getElementById('sendBtn');

            const result = document.getElementById('result');

            const resultMessage = document.getElementById('resultMessage');

            const stats = document.getElementById('stats');

            

            sendBtn.disabled = true;

            sendBtn.textContent = '⏳ در حال ارسال...';

            result.style.display = 'none';

            

            const formData = new FormData();

            formData.append('action', 'send_advertisement');

            formData.append('sender_id', '<?php echo $ADMIN_SENDER_ID; ?>');

            

            if (currentTab === 'message') {

                formData.append('message', document.getElementById('advertisement_message').value);

            } else {

                formData.append('forward_from_chat_id', document.getElementById('forward_from_chat_id').value);

                formData.append('forward_message_id', document.getElementById('forward_message_id').value);

            }

            

            try {

                const response = await fetch('', {

                    method: 'POST',

                    body: formData

                });

                

                const data = await response.json();

                

                if (data.success) {

                    result.className = 'result success';

                    resultMessage.textContent = '✅ تبلیغات با موفقیت ارسال شد!';

                    

                    if (data.stats) {

                        const duration = Math.floor((Date.now() / 1000) - data.stats.start_time);

                        stats.innerHTML = `

                            <div class="stat-card">

                                <div class="stat-number">${data.stats.total_bots}</div>

                                <div class="stat-label">تعداد ربات‌ها</div>

                            </div>

                            <div class="stat-card">

                                <div class="stat-number">${data.stats.total_users}</div>

                                <div class="stat-label">کل کاربران</div>

                            </div>

                            <div class="stat-card">

                                <div class="stat-number">${data.stats.successful_sends}</div>

                                <div class="stat-label">ارسال موفق</div>

                            </div>

                            <div class="stat-card">

                                <div class="stat-number">${data.stats.failed_sends}</div>

                                <div class="stat-label">ارسال ناموفق</div>

                            </div>

                            <div class="stat-card">

                                <div class="stat-number">${duration}s</div>

                                <div class="stat-label">مدت زمان</div>

                            </div>

                        `;

                    }

                } else {

                    result.className = 'result error';

                    resultMessage.textContent = '❌ خطا: ' + data.message;

                    stats.innerHTML = '';

                }

                

                result.style.display = 'block';

                

            } catch (error) {

                result.className = 'result error';

                resultMessage.textContent = '❌ خطا در ارسال درخواست: ' + error.message;

                result.style.display = 'block';

            }

            

            sendBtn.disabled = false;

            sendBtn.textContent = '🚀 شروع ارسال تبلیغات';

        });

    </script>

</body>

</html>

