<?php

namespace RubikaBot;



use PDO; 

use PDOException;



class Database {

	private PDO $pdo;



	public function __construct(string $dbFilePath) {

		$needInit = !file_exists($dbFilePath);

		

		// Ensure the directory exists

        $dbDir = dirname($dbFilePath);

        if (!is_dir($dbDir)) {

            mkdir($dbDir, 0777, true);

        }



		try {

			$this->pdo = new PDO('sqlite:' . $dbFilePath);

			$this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

			// Enable foreign keys

			$this->pdo->exec('PRAGMA foreign_keys = ON');

		} catch (PDOException $e) {

			error_log('Database connection failed: ' . $e->getMessage());

			throw $e;

		}

		

		if ($needInit) {

			error_log("Database: Initializing new series database");

			$this->initSchema();

		}

		// Always run migrations for existing databases

		error_log("Database: Running migrations");

		$this->runMigrations();

	}



	public function getPdo(): PDO { return $this->pdo; }



	private function initSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				chat_id TEXT UNIQUE, 

				first_name TEXT,

				last_name TEXT,

				username TEXT,

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP

			)',

			'CREATE TABLE IF NOT EXISTS series (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				name TEXT UNIQUE, 

				description TEXT,

				poster_url TEXT,

				imdb_rating REAL,

				year INTEGER,

				status TEXT DEFAULT "ongoing",

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP

			)',

			'CREATE TABLE IF NOT EXISTS episodes (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				series_id INTEGER, 

				episode_number INTEGER, 

				title TEXT,

				description TEXT,

				duration INTEGER,

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP, 

				FOREIGN KEY(series_id) REFERENCES series(id) ON DELETE CASCADE, 

				UNIQUE(series_id, episode_number)

			)',

			'CREATE TABLE IF NOT EXISTS qualities (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				episode_id INTEGER, 

				quality TEXT, 

				file_from_chat_id TEXT, 

				file_message_id TEXT, 

				video_from_chat_id TEXT, 

				video_message_id TEXT, 

				file_link TEXT,

				file_size INTEGER,

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP, 

				FOREIGN KEY(episode_id) REFERENCES episodes(id) ON DELETE CASCADE, 

				UNIQUE(episode_id, quality)

			)',

			'CREATE TABLE IF NOT EXISTS user_downloads (

				id INTEGER PRIMARY KEY AUTOINCREMENT,

				user_chat_id TEXT,

				episode_id INTEGER,

				quality_id INTEGER,

				download_date DATETIME DEFAULT CURRENT_TIMESTAMP,

				FOREIGN KEY(episode_id) REFERENCES episodes(id) ON DELETE CASCADE,

				FOREIGN KEY(quality_id) REFERENCES qualities(id) ON DELETE CASCADE

			)',

			'CREATE INDEX IF NOT EXISTS idx_users_chat_id ON users(chat_id)',

			'CREATE INDEX IF NOT EXISTS idx_series_name ON series(name)',

			'CREATE INDEX IF NOT EXISTS idx_episodes_series_id ON episodes(series_id)',

			'CREATE INDEX IF NOT EXISTS idx_qualities_episode_id ON qualities(episode_id)',

			'CREATE INDEX IF NOT EXISTS idx_downloads_user ON user_downloads(user_chat_id)'

		];

		

		foreach ($schema as $sql) {

			try {

				$this->pdo->exec($sql);

				error_log('Database: Created table/index: ' . substr($sql, 0, 50) . '...');

			} catch (PDOException $e) {

				error_log('Database: Error creating table: ' . $e->getMessage());

			}

		}

	}



	private function runMigrations(): void {

		// Migration 1: Add missing columns to qualities table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(qualities)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			$newColumns = [

				'file_from_chat_id' => 'ALTER TABLE qualities ADD COLUMN file_from_chat_id TEXT',

				'file_message_id' => 'ALTER TABLE qualities ADD COLUMN file_message_id TEXT',

				'video_from_chat_id' => 'ALTER TABLE qualities ADD COLUMN video_from_chat_id TEXT',

				'video_message_id' => 'ALTER TABLE qualities ADD COLUMN video_message_id TEXT',

				'file_size' => 'ALTER TABLE qualities ADD COLUMN file_size INTEGER'

			];

			

			foreach ($newColumns as $columnName => $sql) {

				if (!in_array($columnName, $columnNames)) {

					$this->pdo->exec($sql);

					error_log("Database: Added $columnName column to qualities table");

				}

			}

		} catch (PDOException $e) {

			error_log('Database: Migration error for qualities table: ' . $e->getMessage());

		}



		// Migration 2: Add user_downloads table if it doesn't exist

		try {

			$stmt = $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='user_downloads'");

			if (!$stmt->fetch()) {

				$sql = 'CREATE TABLE user_downloads (

					id INTEGER PRIMARY KEY AUTOINCREMENT,

					user_chat_id TEXT,

					episode_id INTEGER,

					quality_id INTEGER,

					download_date DATETIME DEFAULT CURRENT_TIMESTAMP,

					FOREIGN KEY(episode_id) REFERENCES episodes(id) ON DELETE CASCADE,

					FOREIGN KEY(quality_id) REFERENCES qualities(id) ON DELETE CASCADE

				)';

				$this->pdo->exec($sql);

				$this->pdo->exec('CREATE INDEX idx_downloads_user ON user_downloads(user_chat_id)');

				error_log('Database: Created user_downloads table');

			}

		} catch (PDOException $e) {

			error_log('Database: Migration error for user_downloads table: ' . $e->getMessage());

		}

	}



	// Helper methods for series bot functionality

	public function addUser(string $chatId, ?string $firstName = null, ?string $lastName = null, ?string $username = null): bool {

		$stmt = $this->pdo->prepare("INSERT OR IGNORE INTO users (chat_id, first_name, last_name, username) VALUES (?, ?, ?, ?)");

		return $stmt->execute([$chatId, $firstName, $lastName, $username]);

	}



	public function getAllSeries(): array {

		$stmt = $this->pdo->query('SELECT DISTINCT s.id, s.name 

								   FROM series s 

								   JOIN episodes e ON s.id = e.series_id 

								   JOIN qualities q ON e.id = q.episode_id 

								   WHERE (q.file_message_id IS NOT NULL OR q.video_message_id IS NOT NULL)

								   ORDER BY s.name ASC');

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function getSeriesById(int $seriesId): ?array {

		$stmt = $this->pdo->prepare('SELECT * FROM series WHERE id = ?');

		$stmt->execute([$seriesId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function getSeriesByName(string $name): ?array {

		$stmt = $this->pdo->prepare('SELECT * FROM series WHERE name = ?');

		$stmt->execute([$name]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function searchSeries(string $query): array {

		$stmt = $this->pdo->prepare('SELECT id, name FROM series WHERE name LIKE ? ORDER BY name ASC');

		$stmt->execute(["%{$query}%"]);

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function addSeries(string $name, ?string $description = null): bool {

		$stmt = $this->pdo->prepare("INSERT INTO series (name, description) VALUES (?, ?)");

		return $stmt->execute([$name, $description]);

	}



	public function getEpisodes(int $seriesId): array {

		$stmt = $this->pdo->prepare('SELECT DISTINCT e.id, e.episode_number, e.title

									 FROM episodes e 

									 JOIN qualities q ON e.id = q.episode_id 

									 WHERE e.series_id = ? AND (q.file_message_id IS NOT NULL OR q.video_message_id IS NOT NULL)

									 ORDER BY e.episode_number ASC');

		$stmt->execute([$seriesId]);

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function addEpisode(int $seriesId, int $episodeNumber, ?string $title = null): bool {

		$stmt = $this->pdo->prepare("INSERT INTO episodes (series_id, episode_number, title) VALUES (?, ?, ?)");

		return $stmt->execute([$seriesId, $episodeNumber, $title]);

	}



	public function getQualities(int $episodeId): array {

		$stmt = $this->pdo->prepare('SELECT * FROM qualities WHERE episode_id = ? AND (file_message_id IS NOT NULL OR video_message_id IS NOT NULL) ORDER BY quality DESC');

		$stmt->execute([$episodeId]);

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function addOrUpdateQuality(int $episodeId, string $quality, ?string $fileFromChatId = null, ?string $fileMessageId = null, ?string $videoFromChatId = null, ?string $videoMessageId = null): bool {

		// Check if quality exists

		$stmt = $this->pdo->prepare("SELECT id FROM qualities WHERE episode_id = ? AND quality = ?");

		$stmt->execute([$episodeId, $quality]);

		$existingId = $stmt->fetchColumn();



		if ($existingId) {

			// Update existing

			$updates = [];

			$params = [];

			if ($fileFromChatId && $fileMessageId) {

				$updates[] = "file_from_chat_id = ?, file_message_id = ?";

				$params = array_merge($params, [$fileFromChatId, $fileMessageId]);

			}

			if ($videoFromChatId && $videoMessageId) {

				$updates[] = "video_from_chat_id = ?, video_message_id = ?";

				$params = array_merge($params, [$videoFromChatId, $videoMessageId]);

			}

			

			if (!empty($updates)) {

				$sql = "UPDATE qualities SET " . implode(', ', $updates) . " WHERE id = ?";

				$params[] = $existingId;

				$stmt = $this->pdo->prepare($sql);

				return $stmt->execute($params);

			}

		} else {

			// Insert new

			$stmt = $this->pdo->prepare("INSERT INTO qualities (episode_id, quality, file_from_chat_id, file_message_id, video_from_chat_id, video_message_id) VALUES (?, ?, ?, ?, ?, ?)");

			return $stmt->execute([$episodeId, $quality, $fileFromChatId, $fileMessageId, $videoFromChatId, $videoMessageId]);

		}

		

		return false;

	}



	public function logDownload(string $userChatId, int $episodeId, int $qualityId): bool {

		$stmt = $this->pdo->prepare("INSERT INTO user_downloads (user_chat_id, episode_id, quality_id) VALUES (?, ?, ?)");

		return $stmt->execute([$userChatId, $episodeId, $qualityId]);

	}



	public function getStats(): array {

		$stats = [];

		$stats['users'] = $this->pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

		$stats['series'] = $this->pdo->query("SELECT COUNT(*) FROM series")->fetchColumn();

		$stats['episodes'] = $this->pdo->query("SELECT COUNT(*) FROM episodes")->fetchColumn();

		$stats['qualities'] = $this->pdo->query("SELECT COUNT(*) FROM qualities")->fetchColumn();

		$stats['downloads'] = $this->pdo->query("SELECT COUNT(*) FROM user_downloads")->fetchColumn();

		return $stats;

	}

}

?>

