<?php



require_once __DIR__ . '/../../src/RubikaBot.php';

require_once __DIR__ . '/Database.php';



use RubikaBot\Bot;

use RubikaBot\Database;

use RubikaBot\Keypad;

use RubikaBot\Button;

use RubikaBot\KeypadRow;



// --- ⚙️ Configuration ---

$bot = new Bot(BOT_TOKEN);

$db = new Database(DB_FILE_PATH);

$pdo = $db->getPdo();

$STATE_FILE = __DIR__ . '/admin_state.json';

$BOT_STATUS_FILE = __DIR__ . '/bot_status.txt';



// --- 🔄 Bot Status Functions ---

function getBotStatus(): bool {

    global $BOT_STATUS_FILE;

    if (!file_exists($BOT_STATUS_FILE)) {

        file_put_contents($BOT_STATUS_FILE, '1'); // Default: bot is ON

        return true;

    }

    return (bool)trim(file_get_contents($BOT_STATUS_FILE));

}



function setBotStatus(bool $status): void {

    global $BOT_STATUS_FILE;

    file_put_contents($BOT_STATUS_FILE, $status ? '1' : '0');

}



// --- 🔄 State Management Functions ---

function setState(string $chatId, string $action, array $data = []): void {

    global $STATE_FILE;

    $state = ['chat_id' => $chatId, 'action' => $action, 'data' => $data];

    file_put_contents($STATE_FILE, json_encode($state, JSON_PRETTY_PRINT));

}



function getState(): ?array {

    global $STATE_FILE;

    if (!file_exists($STATE_FILE)) return null;

    $content = file_get_contents($STATE_FILE);

    return $content ? json_decode($content, true) : null;

}



function clearState(): void {

    global $STATE_FILE;

    if (file_exists($STATE_FILE)) {

        unlink($STATE_FILE);

    }

}



// --- ℹ️ Get Update Info ---

$update = $bot->getUpdate();

$text = $bot->getText();

$chatId = $bot->getChatId();

$senderId = $bot->getSenderId();

$messageId = $bot->getMessageId();

$buttonId = $bot->getButtonId();



if (!$chatId) {

    error_log("Could not determine chatId from update.");

    exit;

}



// --- 👤 User Tracking ---

try {

    $stmt = $pdo->prepare("INSERT OR IGNORE INTO users (chat_id) VALUES (?)");

    $stmt->execute([$chatId]);

} catch (PDOException $e) {

    error_log("User tracking failed for chat_id {$chatId}: " . $e->getMessage());

}



// --- � Bot Status Check ---

if (!getBotStatus() && $senderId !== ADMIN_SENDER_ID) {

    $bot->chat($chatId)->message("🔐 ربات در حال حاضر خاموش است. لطفاً بعداً تلاش کنید.")->send();

    exit;

}



// --- �👮‍♂️ Admin State Handling ---

$adminState = getState();

if ($senderId === ADMIN_SENDER_ID && $adminState && $adminState['chat_id'] === $chatId && !$buttonId) {

    

    // Check for cancel command first

    if ($text === '❌ لغو' || $text === '/cancel' || $text === 'لغو') {

        clearState();

        $bot->chat($chatId)->message("✅ عملیات لغو شد.")->send();

        send_admin_panel($bot, $chatId);

        exit;

    }

    

    $action = $adminState['action'];

    $data = $adminState['data'];



    switch ($action) {

        case 'waiting_broadcast_message':

            $bot->chat($chatId)->message("⏳ در حال ارسال پیام همگانی... لطفاً صبور باشید.")->send();

            $stmt = $pdo->query("SELECT chat_id FROM users");

            $users = $stmt->fetchAll(PDO::FETCH_COLUMN);

            $sent_count = 0;

            $failed_count = 0;

            foreach ($users as $user_chat_id) {

                try {

                    // To avoid sending to the admin themselves if they are also a user

                    if ($user_chat_id === $chatId) continue;

                    

                    $bot->chat($user_chat_id)->message($text)->send();

                    $sent_count++;

                    usleep(300000); // 0.3 second delay

                } catch (Exception $e) {

                    $failed_count++;

                    error_log("Broadcast failed for user {$user_chat_id}: " . $e->getMessage());

                }

            }

            $bot->chat($chatId)->message("✅ پیام همگانی با موفقیت ارسال شد.\n\n- 📬 ارسال موفق: {$sent_count} کاربر\n- ❌ ارسال ناموفق: {$failed_count} کاربر\n- 👥 کل کاربران: " . count($users))->send();

            clearState();

            send_admin_panel($bot, $chatId);

            exit;

            break;

        case 'waiting_broadcast_forward':

            $forward_message_id = $bot->getMessageId();

            

            $bot->chat($chatId)->message("⏳ در حال فوروارد همگانی... لطفاً صبور باشید.")->send();

            $stmt = $pdo->query("SELECT chat_id FROM users");

            $users = $stmt->fetchAll(PDO::FETCH_COLUMN);

            $sent_count = 0;

            $failed_count = 0;

            foreach ($users as $user_chat_id) {

                try {

                    if ($user_chat_id === $chatId) continue;

                    

                    $bot->forwardFrom($chatId)

                        ->messageId($forward_message_id)

                        ->forwardTo($user_chat_id)

                        ->forward();

                        

                    $sent_count++;

                    usleep(300000); // 0.3 second delay

                } catch (Exception $e) {

                    $failed_count++;

                    error_log("Broadcast forward failed for user {$user_chat_id}: " . $e->getMessage());

                }

            }

            $bot->chat($chatId)->message("✅ فوروارد همگانی با موفقیت انجام شد.\n\n- 📬 ارسال موفق: {$sent_count} کاربر\n- ❌ ارسال ناموفق: {$failed_count} کاربر\n- 👥 کل کاربران: " . count($users))->send();

            clearState();

            send_admin_panel($bot, $chatId);

            exit;



        case 'waiting_series_name':

            try {

                $stmt = $pdo->prepare("INSERT INTO series (name) VALUES (?)");

                $stmt->execute([$text]);

                $bot->chat($chatId)->message("✅ سریال `{$text}` با موفقیت اضافه شد.")->send();

                

                // Go back to the main series management screen

                send_series_management_menu($bot, $pdo, $chatId);



            } catch (PDOException $e) {

                if ($e->errorInfo[1] == 19) { // SQLITE_CONSTRAINT

                     $bot->chat($chatId)->message("⚠️ خطای تکراری! سریالی با نام `{$text}` از قبل وجود دارد.")->send();

                } else {

                     $bot->chat($chatId)->message("❌ خطا در افزودن سریال: " . $e->getMessage())->send();

                }

            }

            clearState();

            exit;

        

        case 'waiting_episode_number':

            $series_id = $data['series_id'];

            if (is_numeric($text)) {

                $episodeNumber = trim($text);

                try {

                    $stmt = $pdo->prepare("INSERT INTO episodes (series_id, episode_number) VALUES (?, ?)");

                    $stmt->execute([$series_id, $episodeNumber]);

                    $bot->chat($chatId)->message("✅ قسمت `{$episodeNumber}` برای سریال با شناسه `{$series_id}` با موفقیت اضافه شد.")->send();

                } catch (PDOException $e) {

                     if ($e->errorInfo[1] == 19) {

                        $bot->chat($chatId)->message("⚠️ خطای تکراری! این قسمت قبلا برای این سریال ثبت شده است.")->send();

                    } else {

                        $bot->chat($chatId)->message("❌ خطا در افزودن قسمت: " . $e->getMessage())->send();

                    }

                }

            } else {

                $bot->chat($chatId)->message("❌ فرمت ورودی نامعتبر است. لطفاً فقط شماره قسمت را ارسال کنید.")->send();

            }

            clearState();

            // After adding, show the series management again as a new message

            send_series_episodes_admin_menu($bot, $pdo, $chatId, $series_id);

            exit;



        case 'waiting_quality_file_forward':

            $episode_id = $data['episode_id'];

            $quality = $data['quality'];

            $type = $data['type']; // 'file' or 'video'

            

            // Check if message has a file

            $file_id = $bot->getFileId();

            if (!$file_id) {

                $bot->chat($chatId)->message("❌ لطفا یک فایل فوروارد کنید، نه متن.")->send();

                exit;

            }

            

            // Get update and check for forward information

            $update = $bot->getUpdate();

            $message = $update['update']['new_message'] ?? null;

            

            // Try to get forwarded info, if not available use current chat

            $from_chat_id = null;

            $forward_message_id = null;

            

            if ($message && isset($message['forwarded_from'])) {

                // Message was forwarded - use original source

                $forwarded_from = $message['forwarded_from'];

                $from_chat_id = $forwarded_from['chat_id'] ?? null;

                $forward_message_id = $forwarded_from['message_id'] ?? null;

            }

            

            // Fallback: if not forwarded, use current message (direct upload)

            if (!$from_chat_id || !$forward_message_id) {

                $from_chat_id = $chatId;  // Admin chat

                $forward_message_id = $bot->getMessageId();  // Current message

            }

            

            try {

                // First, check if a row for this quality already exists

                $stmt = $pdo->prepare("SELECT id FROM qualities WHERE episode_id = ? AND quality = ?");

                $stmt->execute([$episode_id, $quality]);

                $existing_id = $stmt->fetchColumn();



                $from_chat_id_col = "{$type}_from_chat_id";

                $message_id_col = "{$type}_message_id";



                if ($existing_id) {

                    // Update the existing row

                    $stmt = $pdo->prepare("UPDATE qualities SET {$from_chat_id_col} = ?, {$message_id_col} = ? WHERE id = ?");

                    $stmt->execute([$from_chat_id, $forward_message_id, $existing_id]);

                } else {

                    // Insert a new row

                    $stmt = $pdo->prepare("INSERT INTO qualities (episode_id, quality, {$from_chat_id_col}, {$message_id_col}) VALUES (?, ?, ?, ?)");

                    $stmt->execute([$episode_id, $quality, $from_chat_id, $forward_message_id]);

                }

                

                $type_fa = ($type === 'file') ? 'فایل' : 'ویدیو';

                $bot->chat($chatId)->message("✅ کیفیت `{$quality}` با موفقیت برای **{$type_fa}** ثبت شد.")->send();



            } catch (PDOException $e) {

                 $bot->chat($chatId)->message("❌ خطا در افزودن کیفیت: " . $e->getMessage())->send();

            }

            clearState();

            // After adding, show the episode management again as a new message

            send_episode_qualities_admin_menu($bot, $pdo, $chatId, $episode_id);

            exit;



        case 'confirm_replace_quality':

            $episode_id = $data['episode_id'];

            $quality = $data['quality'];

            $from_chat_id = $data['from_chat_id'];

            $message_id = $data['message_id'];

            

            if ($text === '✅ بله، جایگزین کن') {

                try {

                    // Update existing quality

                    $stmt = $pdo->prepare("UPDATE qualities SET from_chat_id = ?, message_id = ? WHERE episode_id = ? AND quality = ?");

                    $stmt->execute([$from_chat_id, $message_id, $episode_id, $quality]);

                    

                    if ($from_chat_id === $chatId) {

                        $bot->chat($chatId)->message("✅ کیفیت `{$quality}` با موفقیت جایگزین شد (فایل مستقیماً آپلود شده).")->send();

                    } else {

                        $bot->chat($chatId)->message("✅ کیفیت `{$quality}` با موفقیت جایگزین شد و برای فوروارد آماده است.")->send();

                    }

                } catch (PDOException $e) {

                    $bot->chat($chatId)->message("❌ خطا در جایگزینی کیفیت: " . $e->getMessage())->send();

                }

                clearState();

                send_episode_qualities_admin_menu($bot, $pdo, $chatId, $episode_id);

            } else {

                // User cancelled

                clearState();

                $bot->chat($chatId)->message("❌ عملیات لغو شد.")->send();

                send_episode_qualities_admin_menu($bot, $pdo, $chatId, $episode_id);

            }

            exit;



        case 'waiting_episode_info':

            $parts = explode(':', $text, 2); // format: series_id:episode_number

            if (count($parts) == 2 && is_numeric(trim($parts[0])) && is_numeric(trim($parts[1]))) {

                $seriesId = trim($parts[0]);

                $episodeNumber = trim($parts[1]);

                try {

                    $stmt = $pdo->prepare("INSERT INTO episodes (series_id, episode_number) VALUES (?, ?)");

                    $stmt->execute([$seriesId, $episodeNumber]);

                    $bot->chat($chatId)->message("✅ قسمت `{$episodeNumber}` برای سریال با شناسه `{$seriesId}` با موفقیت اضافه شد.")->send();

                } catch (PDOException $e) {

                     if ($e->errorInfo[1] == 19) {

                        $bot->chat($chatId)->message("⚠️ خطای تکراری! این قسمت قبلا برای این سریال ثبت شده است.")->send();

                    } else {

                        $bot->chat($chatId)->message("❌ خطا در افزودن قسمت: " . $e->getMessage())->send();

                    }

                }

            } else {

                $bot->chat($chatId)->message("❌ فرمت ورودی نامعتبر است. لطفاً به فرمت `IDسریال:شماره‌قسمت` ارسال کنید. (مثال: `12:5`)")->send();

            }

            clearState();

            send_admin_panel($bot, $chatId);

            exit;



        case 'waiting_quality_info':

            $parts = explode(':', $text, 3); // format: episode_id:quality:file_id

            if (count($parts) == 3 && is_numeric(trim($parts[0]))) {

                $episodeId = trim($parts[0]);

                $quality = trim($parts[1]);

                $fileId = trim($parts[2]);

                try {

                    $stmt = $pdo->prepare("INSERT INTO qualities (episode_id, quality, file_link) VALUES (?, ?, ?)");

                    $stmt->execute([$episodeId, $quality, $fileId]);

                    $bot->chat($chatId)->message("✅ کیفیت `{$quality}` برای قسمت با شناسه `{$episodeId}` با موفقیت اضافه شد.")->send();

                } catch (PDOException $e) {

                    if ($e->errorInfo[1] == 19) {

                        $bot->chat($chatId)->message("⚠️ خطای تکراری! این کیفیت قبلا برای این قسمت ثبت شده است.")->send();

                    } else {

                        $bot->chat($chatId)->message("❌ خطا در افزودن کیفیت: " . $e->getMessage())->send();

                    }

                }

            } else {

                $bot->chat($chatId)->message("❌ فرمت ورودی نامعتبر است. لطفاً به فرمت `IDقسمت:کیفیت:فایل‌آیدی` ارسال کنید. (مثال: `45:720p:file_id_string`)")->send();

            }

            clearState();

            send_admin_panel($bot, $chatId);

            exit;

    }

}





// --- 💡 Main Logic ---

if ($buttonId) {

    // This block now ONLY handles INLINE buttons (buttons with callback data)

    $parts = explode('_', $buttonId);

    $action = $parts[0] ?? null;



    // Admin inline button handling (mostly for delete operations)

    if ($senderId === ADMIN_SENDER_ID && $action === 'admin') {

        handle_admin_inline_buttons($bot, $pdo, $chatId, $messageId, $parts);

        exit;

    }



    // User inline button handling

    switch ($action) {

        case 'episode':

            $episodeId = $parts[1] ?? null;

            if ($episodeId && $messageId) {

                // مرحله ۱: نمایش منوی انتخاب کیفیت

                send_qualities_menu($bot, $pdo, $chatId, $messageId, $episodeId);

            }

            break;



        case 'quality':

            $qualityId = $parts[1] ?? null;

            if ($qualityId && $messageId) {

                // مرحله ۲: پس از انتخاب کیفیت، نمایش منوی نوع فایل

                send_file_type_menu($bot, $pdo, $chatId, $messageId, $qualityId);

            }

            break;



        case 'download':

            $qualityId = $parts[1] ?? null;

            $type = $parts[2] ?? null; // 'file' or 'video'



            if ($qualityId && $type) {

                $from_chat_id_col = "{$type}_from_chat_id";

                $message_id_col = "{$type}_message_id";



                $stmt = $pdo->prepare("SELECT q.{$from_chat_id_col} as from_chat_id, q.{$message_id_col} as message_id, q.quality, e.episode_number, s.name as series_name 

                                       FROM qualities q

                                       JOIN episodes e ON q.episode_id = e.id

                                       JOIN series s ON e.series_id = s.id

                                       WHERE q.id = ? AND q.{$from_chat_id_col} IS NOT NULL AND q.{$message_id_col} IS NOT NULL");

                $stmt->execute([$qualityId]);

                $result = $stmt->fetch();

                if ($result) {

                    try {

                        // Forward the file from admin chat to user

                        $bot->forwardFrom($result['from_chat_id'])

                            ->messageId($result['message_id'])

                            ->forwardTo($chatId)

                            ->forward();

                        

                        // Send a caption message after forwarding

                        $type_fa = ($type === 'file') ? 'فایل' : 'ویدیو';

                        $caption = "🎬 سریال: {$result['series_name']}\n" .

                                   "🍿 قسمت: {$result['episode_number']}\n" .

                                   "🖥 کیفیت: {$result['quality']} ({$type_fa})\n\n" .

                                   "✨ @MegaAboli7-Bot"; // <-- آیدی کانال یا ربات خود را اینجا قرار دهید

                        $bot->chat($chatId)->message($caption)->send();

                    } catch (Exception $e) {

                        $bot->chat($chatId)->message('❌ خطا در ارسال فایل. لطفاً دوباره تلاش کنید.')->send();

                        error_log("Forward failed: " . $e->getMessage());

                    }

                } else {

                    $bot->chat($chatId)->message('❌ خطا: فایل مورد نظر یافت نشد.')->send();

                }

            }

            break;

        case 'back':

             $target = $parts[1] ?? 'main';

             if ($target === 'main' && $messageId) {

                 try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch(Exception $e) {}

                 send_main_menu($bot, $pdo, $chatId);



             } else if ($target === 'series' && isset($parts[2]) && $messageId) {

                 $seriesId = $parts[2];

                 $stmt = $pdo->prepare('SELECT name FROM series WHERE id = ?');

                 $stmt->execute([$seriesId]);

                 $seriesName = $stmt->fetchColumn();

                 try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch(Exception $e) {}

                 send_episodes_menu($bot, $pdo, $chatId, $seriesId, "🎬 سریال " . $seriesName . " - ");

             }

             break;

    }



} else if ($text) {

    

    // Admin text-based command handling

    if ($senderId === ADMIN_SENDER_ID) {

        $admin_action = get_admin_action_from_text($text);

        if ($admin_action) {

            handle_admin_text_buttons($bot, $pdo, $chatId, $admin_action);

            exit;

        }

    }



    // General user text-based command handling

    switch (true) {

        case in_array(strtolower($text), ['/start', 'start', 'شروع', '🏠 بازگشت به منوی اصلی']):

            send_main_menu($bot, $pdo, $chatId);

            break;

        

        case strpos($text, '🎬') === 0:

            $seriesName = trim(str_replace('🎬', '', $text));

            $stmt = $pdo->prepare('SELECT id FROM series WHERE name = ?');

            $stmt->execute([$seriesName]);

            $seriesId = $stmt->fetchColumn();

            if ($seriesId) {

                send_episodes_menu($bot, $pdo, $chatId, $seriesId, "🎬 سریال " . $seriesName . " - ");

            } else {

                send_main_menu($bot, $pdo, $chatId, "🤷‍♂️ متاسفانه سریالی با نام `{$seriesName}` یافت نشد.");

            }

            break;



        default:

            // Search for series by name (when user types it)

            $stmt = $pdo->prepare('SELECT id, name FROM series WHERE name LIKE ?');

            $stmt->execute(["%{$text}%"]);

            $results = $stmt->fetchAll();

            

            if (count($results) === 1) {

                $series = $results[0];

                send_episodes_menu($bot, $pdo, $chatId, $series['id'], "🎬 سریال " . $series['name'] . " - ");

            } else if (count($results) > 1) {

                $keypad = Keypad::make()->setResize(true);

                foreach ($results as $s) {

                    $keypad->row()->add(Button::simple('text_button', '🎬 ' . $s['name']));

                }

                $bot->chat($chatId)->message('🔎 چندین نتیجه برای جستجوی شما یافت شد. لطفاً یکی را انتخاب کنید:')->chatKeypad($keypad->toArray())->send();

            } else {

                send_main_menu($bot, $pdo, $chatId, "🤷‍♂️ متاسفانه سریالی با نام `{$text}` یافت نشد. لطفاً از لیست زیر انتخاب کنید:");

            }

            break;

    }

}



// ---  Functions ---



function get_admin_action_from_text(string $text): ?string {

    $map = [

        '/panel' => 'panel',

        '👑 پنل مدیریت' => 'panel',

        '📊 آمار ربات' => 'stats',

        '📢 پیام همگانی' => 'broadcast',

        '📤 فوروارد همگانی' => 'broadcast_forward',

        '🗂️ مدیریت سریال‌ها' => 'manage_series',

        '➕ سریال جدید' => 'add_series',

        '🔴 خاموش کردن ربات' => 'bot_off',

        '🟢 روشن کردن ربات' => 'bot_on',

        '⬅️ بازگشت به پنل' => 'panel',

        '🏠 بازگشت به منوی اصلی' => 'back_to_main',

    ];

    return $map[$text] ?? null;

}



function send_main_menu(Bot $bot, PDO $pdo, string $chatId, string $message = '👋 سلام! به ربات دانلود سریال خوش آمدید.\n\n🍿 لطفا سریال مورد نظر خود را از لیست زیر انتخاب کنید یا نام آن را جستجو کنید:'): void {

    

    

    // Only show series that have episodes with available files (either file or video)

    $stmt = $pdo->query('SELECT DISTINCT s.id, s.name 

                         FROM series s 

                         JOIN episodes e ON s.id = e.series_id 

                         JOIN qualities q ON e.id = q.episode_id 

                         WHERE (q.file_message_id IS NOT NULL OR q.video_message_id IS NOT NULL)

                         ORDER BY s.name ASC');

    $series = $stmt->fetchAll();



    if ($series) {

        $keypad = Keypad::make()->setResize(true);

        $row = null;

        foreach ($series as $s) {

            // Displaying a limited number of series per row for better UI

            if ($row === null || count($row->toArray()['buttons']) >= 2) {

                $row = $keypad->row();

            }

            $row->add(Button::simple('text_button', '🎬 ' . $s['name']));

        }

        

        // Add admin panel button if user is admin

        global $senderId;

        if ($senderId === ADMIN_SENDER_ID) {

            $keypad->row()->add(Button::simple('👑 پنل مدیریت', '👑 پنل مدیریت'));

        }

        

        $bot->chat($chatId)->message($message)->chatKeypad($keypad->toArray())->send();

    } else {

        $keypad = Keypad::make()->setResize(true);

        // Add admin panel button even if no series exist

        global $senderId;

        if ($senderId === ADMIN_SENDER_ID) {

            $keypad->row()->add(Button::simple('👑 پنل مدیریت', '👑 پنل مدیریت'));

            $bot->chat($chatId)->message('سلام! 😕 هنوز هیچ سریالی با فایل آماده در ربات وجود ندارد. به زودی با کلی سریال برمیگردیم!')->chatKeypad($keypad->toArray())->send();

        } else {

            $bot->chat($chatId)->message('سلام! 😕 هنوز هیچ سریالی با فایل آماده در ربات وجود ندارد. به زودی با کلی سریال برمیگردیم!')->send();

        }

    }

}



function send_episodes_menu(Bot $bot, PDO $pdo, string $chatId, int $seriesId, string $messagePrefix = ''): void {

    // Only show episodes that have at least one quality with valid files (either file or video)

    $stmt = $pdo->prepare('SELECT DISTINCT e.id, e.episode_number 

                           FROM episodes e 

                           JOIN qualities q ON e.id = q.episode_id 

                           WHERE e.series_id = ? AND (q.file_message_id IS NOT NULL OR q.video_message_id IS NOT NULL)

                           ORDER BY e.episode_number ASC');

    $stmt->execute([$seriesId]);

    $episodes = $stmt->fetchAll();



    if ($episodes) {

        $inlineKeypad = Keypad::make();

        $row = new KeypadRow();

        $count = 0;

        foreach ($episodes as $episode) {

            $row->add(Button::simple('episode_' . $episode['id'], "🍿 قسمت " . (string)$episode['episode_number']));

            $count++;

            if ($count % 5 == 0) {

                $inlineKeypad->addRow($row);

                $row = new KeypadRow();

            }

        }

        if ($count > 0 && $count % 5 != 0) {

            $inlineKeypad->addRow($row);

        }

        

        

        $bot->chat($chatId)->message($messagePrefix . 'لطفا قسمت مورد نظر را انتخاب کنید:')->inlineKeypad($inlineKeypad->toArray())->send();

    } else {

        $bot->chat($chatId)->message('😕 برای این سریال هنوز قسمتی با فایل آماده وجود ندارد.')->send();

    }

}



function send_episode_quality_selection_prompt(Bot $bot, PDO $pdo, string $chatId, string $messageId, int $episodeId): void {

    // This function is now effectively replaced by send_qualities_menu, 

    // but we keep it to avoid breaking old references. It will just call the new function.

    send_qualities_menu($bot, $pdo, $chatId, $messageId, $episodeId);

}



function send_qualities_menu(Bot $bot, PDO $pdo, string $chatId, string $messageId, int $episodeId): void {

    $stmt = $pdo->prepare('SELECT q.id, q.quality, e.series_id, s.name as series_name, e.episode_number

                           FROM qualities q 

                           JOIN episodes e ON q.episode_id = e.id

                           JOIN series s ON e.series_id = s.id

                           WHERE q.episode_id = ? AND (q.file_message_id IS NOT NULL OR q.video_message_id IS NOT NULL)

                           ORDER BY q.quality DESC');

    $stmt->execute([$episodeId]);

    $qualities = $stmt->fetchAll();



    if ($qualities) {

        $inlineKeypad = Keypad::make();

        

        $row = new KeypadRow();

        $count = 0;

        foreach ($qualities as $quality) {

            // The button now leads to the file type selection

            $row->add(Button::simple("quality_{$quality['id']}", "💎 {$quality['quality']}"));

            $count++;

            if ($count % 2 == 0) { // 2 qualities per row

                $inlineKeypad->addRow($row);

                $row = new KeypadRow();

            }

        }

        if ($count > 0 && $count % 2 != 0) {

            $inlineKeypad->addRow($row);

        }

        





        $seriesName = $qualities[0]['series_name'] ?? '';

        $episodeNumber = $qualities[0]['episode_number'] ?? '';

        $messageText = "🎬 **{$seriesName}** - قسمت **{$episodeNumber}**\n\n💎 لطفا کیفیت مورد نظر را انتخاب کنید:";



        // To prevent errors, we delete the message and then send the new menu.

        try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch(Exception $e) {}

        $bot->chat($chatId)->message($messageText)->inlineKeypad($inlineKeypad->toArray())->send();



    } else {

        $bot->chat($chatId)->message('😕 برای این قسمت هنوز کیفیتی ثبت نشده است.')->send();

    }

}



function send_file_type_menu(Bot $bot, PDO $pdo, string $chatId, string $messageId, int $qualityId): void {

    $stmt = $pdo->prepare('SELECT q.quality, q.file_message_id, q.video_message_id, e.id as episode_id, e.episode_number, s.name as series_name

                           FROM qualities q

                           JOIN episodes e ON q.episode_id = e.id

                           JOIN series s ON e.series_id = s.id

                           WHERE q.id = ?');

    $stmt->execute([$qualityId]);

    $quality = $stmt->fetch();



    if ($quality) {

        $inlineKeypad = Keypad::make();

        $row = $inlineKeypad->row();

        

        $has_file = false;

        if (!empty($quality['file_message_id'])) {

            $row->add(Button::simple("download_{$qualityId}_file", '📁 فایل'));

            $has_file = true;

        }

        if (!empty($quality['video_message_id'])) {

            $row->add(Button::simple("download_{$qualityId}_video", '▶️ ویدیو'));

            $has_file = true;

        }



        if (!$has_file) {

             $bot->chat($chatId)->message('❌ متاسفانه فایلی برای این کیفیت یافت نشد.')->send();

             return;

        }



        



        $seriesName = $quality['series_name'] ?? '';

        $episodeNumber = $quality['episode_number'] ?? '';

        $qualityName = $quality['quality'] ?? '';

        $messageText = "🎬 **{$seriesName}** - قسمت **{$episodeNumber}** - کیفیت **{$qualityName}**\n\n📁 لطفا نوع فایل را انتخاب کنید:";



        // To prevent errors, we delete the message and then send the new menu.

        try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch(Exception $e) {}

        $bot->chat($chatId)->message($messageText)->inlineKeypad($inlineKeypad->toArray())->send();



    } else {

        $bot->chat($chatId)->message('❌ خطا: کیفیت مورد نظر یافت نشد.')->send();

    }

}





function send_admin_panel(Bot $bot, string $chatId): void {

    $bot_status = getBotStatus();

    $status_text = $bot_status ? "🟢 روشن" : "🔴 خاموش";

    $toggle_button = $bot_status ? "🔴 خاموش کردن ربات" : "🟢 روشن کردن ربات";

    

    $keypad = Keypad::make()->setResize(true);

    

    // Header with bot status

    $header_message = "👑 **پنل مدیریت ربات**\n\n" .

                      "🤖 وضعیت ربات: {$status_text}\n" .

                      "⚡️ برای مدیریت کامل ربات از دکمه‌های زیر استفاده کنید:\n\n";

    

    // Row 1: Bot control

    $keypad->row()

        ->add(Button::simple($toggle_button, $toggle_button));

    

    // Row 2: Statistics and Management

    $keypad->row()

        ->add(Button::simple('📊 آمار ربات', '📊 آمار ربات'))

        ->add(Button::simple('🗂️ مدیریت سریال‌ها', '🗂️ مدیریت سریال‌ها'));

    

    // Row 3: Broadcasting features

    $keypad->row()

        ->add(Button::simple('📢 پیام همگانی', '📢 پیام همگانی'))

        ->add(Button::simple('📤 فوروارد همگانی', '📤 فوروارد همگانی'));

    

    // Row 4: Back button

    $keypad->row()

        ->add(Button::simple('🏠 بازگشت به منوی اصلی', '🏠 بازگشت به منوی اصلی'));

    

    $bot->chat($chatId)->message($header_message)->chatKeypad($keypad->toArray())->send();

}



function handle_admin_text_buttons(Bot $bot, PDO $pdo, string $chatId, string $action): void {

    switch ($action) {

        case 'panel':

            clearState();

            send_admin_panel($bot, $chatId);

            break;

        case 'stats':

            $user_count = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

            $series_count = $pdo->query("SELECT COUNT(*) FROM series")->fetchColumn();

            $episodes_count = $pdo->query("SELECT COUNT(*) FROM episodes")->fetchColumn();

            $files_count = $pdo->query("SELECT COUNT(*) FROM qualities")->fetchColumn();

            $stats_message = "📊 **آمار ربات شما:**\n\n" .

                             "👤 **تعداد کاربران:** {$user_count} نفر\n" .

                             "🎬 **تعداد سریال‌ها:** {$series_count} عنوان\n" .

                             "🍿 **تعداد قسمت‌ها:** {$episodes_count} قسمت\n" .

                             "📁 **تعداد فایل‌ها:** {$files_count} فایل";

            $bot->chat($chatId)->message($stats_message)->send();

            break;

        case 'broadcast':

            setState($chatId, 'waiting_broadcast_message');

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("📝 لطفا پیامی که می‌خواهید برای تمام کاربران ارسال شود را بنویسید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;

        

        case 'manage_series':

            send_series_management_menu($bot, $pdo, $chatId);

            break;



        case 'add_series':

            setState($chatId, 'waiting_series_name');

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("✍️ لطفا نام سریال جدید را ارسال کنید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;

            

        case 'broadcast_forward':

            setState($chatId, 'waiting_broadcast_forward');

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("📤 لطفا پیامی که می‌خواهید برای تمام کاربران فوروارد شود را ارسال کنید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;

            

        case 'bot_off':

            setBotStatus(false);

            $bot->chat($chatId)->message("🔴 ربات با موفقیت خاموش شد. تنها شما به عنوان ادمین می‌توانید از ربات استفاده کنید.")->send();

            send_admin_panel($bot, $chatId);

            break;

            

        case 'bot_on':

            setBotStatus(true);

            $bot->chat($chatId)->message("🟢 ربات با موفقیت روشن شد. تمام کاربران می‌توانند از ربات استفاده کنند.")->send();

            send_admin_panel($bot, $chatId);

            break;

            

        case 'back_to_main':

            clearState();

            send_main_menu($bot, $pdo, $chatId);

            break;

    }

}



function send_series_management_menu(Bot $bot, PDO $pdo, string $chatId, ?string $messageId = null, string $messageText = "🗂️ **مدیریت سریال‌ها**\n\nیک سریال را برای مدیریت انتخاب کنید یا یک سریال جدید اضافه کنید:"): void {

    $stmt = $pdo->query("SELECT id, name FROM series ORDER BY name ASC");

    $series_list = $stmt->fetchAll();

    

    $inlineKeypad = Keypad::make();

    foreach ($series_list as $s) {

        $inlineKeypad->row()->add(Button::simple("admin_show_series_{$s['id']}", '🎬 ' . $s['name']));

    }

    $inlineKeypad->row()->add(Button::simple("admin_add_series_prompt", '➕ افزودن سریال جدید'));

    $inlineKeypad->row()->add(Button::simple("admin_panel", '⬅️ بازگشت به پنل'));



    // Always send as new message instead of edit

    $bot->chat($chatId)->message($messageText)->inlineKeypad($inlineKeypad->toArray())->send();

}



function send_series_episodes_admin_menu(Bot $bot, PDO $pdo, string $chatId, int $series_id, ?string $messageId = null): void {

    $stmt = $pdo->prepare("SELECT name FROM series WHERE id = ?");

    $stmt->execute([$series_id]);

    $series_name = $stmt->fetchColumn();



    if (!$series_name) {

        $bot->chat($chatId)->message("خطا: سریال یافت نشد.")->send();

        send_series_management_menu($bot, $pdo, $chatId);

        return;

    }



    $stmt = $pdo->prepare("SELECT id, episode_number FROM episodes WHERE series_id = ? ORDER BY episode_number ASC");

    $stmt->execute([$series_id]);

    $episodes = $stmt->fetchAll();



    $inlineKeypad = Keypad::make();

    $inlineKeypad->row()->add(Button::simple("admin_add_episode_prompt_{$series_id}", '➕ افزودن قسمت جدید'));

    

    foreach($episodes as $episode) {

        $inlineKeypad->row()->add(Button::simple("admin_show_episode_{$episode['id']}", "🍿 قسمت {$episode['episode_number']}"))

                         ->add(Button::simple("admin_confirm_delete_episode_{$episode['id']}", '🗑'));

    }



    $inlineKeypad->row()->add(Button::simple("admin_confirm_delete_series_{$series_id}", '❌ حذف کل این سریال'));

    $inlineKeypad->row()->add(Button::simple("admin_manage_series", '⬅️ بازگشت به لیست سریال‌ها'));

    

    $messageText = "🎬 مدیریت سریال: **{$series_name}**";



    // Always send as new message instead of edit

    $bot->chat($chatId)->message($messageText)->inlineKeypad($inlineKeypad->toArray())->send();

}



function send_episode_qualities_admin_menu(Bot $bot, PDO $pdo, string $chatId, int $episode_id, ?string $messageId = null): void {

     $stmt = $pdo->prepare("

        SELECT e.episode_number, s.name as series_name, s.id as series_id

        FROM episodes e

        JOIN series s ON e.series_id = s.id

        WHERE e.id = ?

    ");

    $stmt->execute([$episode_id]);

    $episode_info = $stmt->fetch();



    if (!$episode_info) {

        $bot->chat($chatId)->message("خطا: قسمت یافت نشد.")->send();

        return;

    }



    $stmt = $pdo->prepare("SELECT id, quality, file_message_id, video_message_id FROM qualities WHERE episode_id = ?");

    $stmt->execute([$episode_id]);

    $qualities = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $qualities_by_name = array_column($qualities, null, 'quality');



    $defined_qualities = ['360p', '480p', '720p', '1080p'];

    $inlineKeypad = Keypad::make();



    foreach ($defined_qualities as $q) {

        $row = $inlineKeypad->row();

        $row->add(Button::simple("admin_dummy_button", "💎 {$q}")); // Dummy button as a label



        $quality_data = $qualities_by_name[$q] ?? null;



        // --- File Button ---

        if ($quality_data && !empty($quality_data['file_message_id'])) {

            $row->add(Button::simple("admin_edit_quality_{$quality_data['id']}_file", "✅ فایل"));

        } else {

            $row->add(Button::simple("admin_add_quality_prompt_{$episode_id}_{$q}_file", "➕ فایل"));

        }



        // --- Video Button ---

        if ($quality_data && !empty($quality_data['video_message_id'])) {

            $row->add(Button::simple("admin_edit_quality_{$quality_data['id']}_video", "✅ ویدیو"));

        } else {

            $row->add(Button::simple("admin_add_quality_prompt_{$episode_id}_{$q}_video", "➕ ویدیو"));

        }

    }

    

    $inlineKeypad->row()->add(Button::simple("admin_show_series_{$episode_info['series_id']}", '⬅️ بازگشت به قسمت‌ها'));

    $messageText = "🎬 سریال: **{$episode_info['series_name']}**\n🍿 مدیریت قسمت: **{$episode_info['episode_number']}**\n\n- ✅: ویرایش/حذف کیفیت\n- ➕: افزودن کیفیت";



    // Always send as new message instead of edit

    $bot->chat($chatId)->message($messageText)->inlineKeypad($inlineKeypad->toArray())->send();

}



function handle_admin_inline_buttons(Bot $bot, PDO $pdo, string $chatId, ?string $messageId, array $parts): void {

    if (!$messageId) {

        $bot->chat($chatId)->message("خطا: اطلاعات پیام برای ویرایش یافت نشد. لطفاً از پنل اصلی دوباره شروع کنید.")->send();

        send_admin_panel($bot, $chatId);

        return;

    }

    

    $original_parts = $parts; // Keep a copy of the original parts

    $sub_action = null;

    $id = null;

    $type_suffix = null;



    // --- Determine Action ---

    // Priority 1: Check for the most specific pattern first.

    // admin_add_quality_prompt_{episode_id}_{quality}_{type}

    if (count($parts) === 7 && $parts[1] === 'add' && $parts[2] === 'quality' && $parts[3] === 'prompt') {

        $sub_action = 'add_quality_prompt';

    } 

    // Priority 2: Check for patterns ending in _file or _video

    // e.g., admin_edit_quality_123_file

    else if (count($parts) > 3 && in_array(end($parts), ['file', 'video'])) {

        $type_suffix = array_pop($parts);

        $id = array_pop($parts);

        $sub_action = implode('_', array_slice($parts, 1));

    } 

    // Priority 3: Generic case for other buttons with a numeric ID at the end.

    // e.g., admin_show_series_123

    else if (count($parts) > 2 && is_numeric(end($parts))) {

        $id = array_pop($parts);

        $sub_action = implode('_', array_slice($parts, 1));

    }

    // Fallback for simple buttons without IDs

    else {

        $sub_action = implode('_', array_slice($parts, 1));

    }

 

    switch ($sub_action) {

        case 'panel':

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_admin_panel($bot, $chatId);

            break;

            

        case 'toggle_bot':

            $current_status = getBotStatus();

            setBotStatus(!$current_status);

            $new_status = !$current_status;

            $status_message = $new_status ? 

                "🟢 ربات با موفقیت روشن شد. تمام کاربران می‌توانند از ربات استفاده کنند." : 

                "🔴 ربات با موفقیت خاموش شد. تنها شما به عنوان ادمین می‌توانید از ربات استفاده کنید.";

            

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $bot->chat($chatId)->message($status_message)->send();

            send_admin_panel($bot, $chatId);

            break;

            

        case 'stats':

            $user_count = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

            $series_count = $pdo->query("SELECT COUNT(*) FROM series")->fetchColumn();

            $episodes_count = $pdo->query("SELECT COUNT(*) FROM episodes")->fetchColumn();

            $files_count = $pdo->query("SELECT COUNT(*) FROM qualities")->fetchColumn();

            $bot_status = getBotStatus() ? "🟢 روشن" : "🔴 خاموش";

            

            $stats_message = "📊 **آمار کامل ربات شما:**\n\n" .

                             "🤖 **وضعیت ربات:** {$bot_status}\n" .

                             "👤 **تعداد کاربران:** {$user_count} نفر\n" .

                             "🎬 **تعداد سریال‌ها:** {$series_count} عنوان\n" .

                             "🍿 **تعداد قسمت‌ها:** {$episodes_count} قسمت\n" .

                             "📁 **تعداد فایل‌ها:** {$files_count} فایل\n\n" .

                             "⚡️ ربات شما عالی کار می‌کند!";

            

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $bot->chat($chatId)->message($stats_message)->send();

            send_admin_panel($bot, $chatId);

            break;

            

        case 'broadcast':

            setState($chatId, 'waiting_broadcast_message');

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("📝 لطفا پیامی که می‌خواهید برای تمام کاربران ارسال شود را بنویسید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;

            

        case 'broadcast_forward':

            setState($chatId, 'waiting_broadcast_forward');

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("📤 لطفا پیامی که می‌خواهید برای تمام کاربران فوروارد شود را ارسال کنید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;

            

        case 'manage_series_btn':

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_series_management_menu($bot, $pdo, $chatId);

            break;

            

        case 'back_to_main':

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_main_menu($bot, $pdo, $chatId);

            break;



        case 'add_series_prompt':

            setState($chatId, 'waiting_series_name');

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("✍️ لطفا نام سریال جدید را ارسال کنید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;



        case 'show_series':

            $series_id = $id;

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_series_episodes_admin_menu($bot, $pdo, $chatId, $series_id);

            break;



        case 'manage_series':

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_series_management_menu($bot, $pdo, $chatId);

            break;



        case 'add_episode_prompt':

            $series_id = $id;

            setState($chatId, 'waiting_episode_number', ['series_id' => $series_id]);

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("✍️ لطفا شماره قسمت جدید را برای این سریال ارسال کنید:")->chatKeypad($cancelKeypad->toArray())->send();

            break;



        case 'show_episode':

            $episode_id = $id;

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_episode_qualities_admin_menu($bot, $pdo, $chatId, $episode_id);

            break;



        case 'add_quality_prompt':

            // Use the original parts array as it was not modified

            $episode_id = $original_parts[4];

            $quality = $original_parts[5];

            $type = $original_parts[6]; // 'file' or 'video'

            $type_fa = ($type === 'file') ? 'فایل' : 'ویدیو';



            setState($chatId, 'waiting_quality_file_forward', ['episode_id' => $episode_id, 'quality' => $quality, 'type' => $type]);

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $cancelKeypad = Keypad::make()->setResize(true);

            $cancelKeypad->row()->add(Button::simple('❌ لغو', '❌ لغو'));

            $bot->chat($chatId)->message("📤 لطفا **{$type_fa}** مربوط به کیفیت **{$quality}** را ارسال کنید:\n\n🔄 **فوروارد کنید** (بهترین روش)\n📁 یا مستقیماً آپلود کنید")->chatKeypad($cancelKeypad->toArray())->send();

            break;

        

        case 'edit_quality':

            $quality_id = $id;

            $type = $type_suffix; // file or video

            $type_fa = ($type === 'file') ? 'فایل' : 'ویدیو';



            $stmt = $pdo->prepare("SELECT quality FROM qualities WHERE id = ?");

            $stmt->execute([$quality_id]);

            $quality_name = $stmt->fetchColumn();



            $inlineKeypad = Keypad::make();

            $inlineKeypad->row()->add(Button::simple("admin_confirm_delete_quality_{$quality_id}_{$type}", "🗑 حذف {$type_fa}"));

            $inlineKeypad->row()->add(Button::simple("admin_show_episode_for_quality_{$quality_id}", '⬅️ بازگشت'));

            

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $bot->chat($chatId)->message("مدیریت **{$type_fa}** کیفیت **{$quality_name}**")->inlineKeypad($inlineKeypad->toArray())->send();

            break;

        

        case 'show_episode_for_quality':

            $quality_id = $id;

            $stmt = $pdo->prepare("SELECT episode_id FROM qualities WHERE id = ?");

            $stmt->execute([$quality_id]);

            $episode_id = $stmt->fetchColumn();

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            send_episode_qualities_admin_menu($bot, $pdo, $chatId, $episode_id);

            break;



        case 'confirm_delete_series':

        case 'confirm_delete_episode':

        case 'confirm_delete_quality':

            $type = str_replace(['confirm_delete_'], '', $sub_action);

            $item_id = $id;

            $delete_type = $type_suffix; // file or video, if present



            // Determine the correct "back" button

            $back_button_callback = 'admin_manage_series'; // Default

            if ($type === 'quality') {

                $stmt = $pdo->prepare("SELECT episode_id FROM qualities WHERE id = ?");

                $stmt->execute([$item_id]);

                $episode_id = $stmt->fetchColumn();

                if ($episode_id) {

                    $back_button_callback = "admin_show_episode_{$episode_id}";

                }

            } else if ($type === 'episode') {

                $stmt = $pdo->prepare("SELECT series_id FROM episodes WHERE id = ?");

                $stmt->execute([$item_id]);

                $series_id = $stmt->fetchColumn();

                if ($series_id) {

                    $back_button_callback = "admin_show_series_{$series_id}";

                }

            }

            

            $delete_callback = "admin_do_delete_{$type}_{$item_id}";

            if ($delete_type) {

                $delete_callback .= "_{$delete_type}";

            }



            $inlineKeypad = Keypad::make();

            $inlineKeypad->row()

                ->add(Button::simple($delete_callback, '🗑 بله، حذف کن'))

                ->add(Button::simple($back_button_callback, ' خیر، منصرف شدم'));

            

            if ($messageId) {

                try { $bot->chat($chatId)->messageId($messageId)->sendDelete(); } catch (Exception $e) {}

            }

            $bot->chat($chatId)->message("❓ آیا از حذف این آیتم مطمئن هستید؟ این عمل غیرقابل بازگشت است.")->inlineKeypad($inlineKeypad->toArray())->send();

            break;



        case 'do_delete_series':

        case 'do_delete_episode':

        case 'do_delete_quality':

            $type = str_replace(['do_delete_'], '', $sub_action);

            $item_id = $id;

            $delete_type = $type_suffix; // file or video, if present



            // Correctly map type to table name

            $table_name = '';

            if ($type === 'series') {

                $table_name = 'series';

            } else if ($type === 'episode') {

                $table_name = 'episodes';

            } else if ($type === 'quality') {

                $table_name = 'qualities';

            }



            if (empty($table_name)) {

                $bot->chat($chatId)->message("❌ خطای داخلی: نوع حذف نامعتبر است.")->send();

                send_series_management_menu($bot, $pdo, $chatId);

                break;

            }



            // --- Find parent ID before deleting ---

            $parent_series_id = null;

            $parent_episode_id = null;



            if ($type === 'quality') {

                $stmt = $pdo->prepare("SELECT e.id as episode_id, e.series_id FROM qualities q JOIN episodes e ON q.episode_id = e.id WHERE q.id = ?");

                $stmt->execute([$item_id]);

                $res = $stmt->fetch();

                if ($res) {

                    $parent_episode_id = $res['episode_id'];

                    $parent_series_id = $res['series_id'];

                }

            } else if ($type === 'episode') {

                $stmt = $pdo->prepare("SELECT series_id FROM episodes WHERE id = ?");

                $stmt->execute([$item_id]);

                $parent_series_id = $stmt->fetchColumn();

            }

            // For series, there is no parent in this context.



            try {

                if ($type === 'quality' && $delete_type) {

                    // Just nullify the specific type (file or video), don't delete the row

                    $column_to_null_prefix = ($delete_type === 'file') ? 'file' : 'video';

                    $stmt = $pdo->prepare("UPDATE qualities SET {$column_to_null_prefix}_from_chat_id = NULL, {$column_to_null_prefix}_message_id = NULL WHERE id = ?");

                    $stmt->execute([$item_id]);

                    $bot->chat($chatId)->message("✅ نوع `{$delete_type}` برای این کیفیت با موفقیت حذف شد.")->send();



                } else {

                    // Delete the entire row for series, episode, or if quality type is not specified

                    $stmt = $pdo->prepare("DELETE FROM {$table_name} WHERE id = ?");

                    $stmt->execute([$item_id]);

                    if ($stmt->rowCount() > 0) {

                        $bot->chat($chatId)->message("✅ آیتم از جدول `{$table_name}` با شناسه `{$item_id}` با موفقیت حذف شد.")->send();

                    } else {

                        $bot->chat($chatId)->message("⚠️ آیتم مورد نظر یافت نشد یا قبلا حذف شده است.")->send();

                    }

                }

            } catch (PDOException $e) {

                $bot->chat($chatId)->message("❌ خطا در حذف: " . $e->getMessage())->send();

            }



            // --- Go back to the correct parent menu ---

            if ($type === 'quality' && $parent_episode_id) {

                send_episode_qualities_admin_menu($bot, $pdo, $chatId, $parent_episode_id);

            } else if ($type === 'episode' && $parent_series_id) {

                send_series_episodes_admin_menu($bot, $pdo, $chatId, $parent_series_id);

            } else { // This covers series deletion and any fallbacks

                send_series_management_menu($bot, $pdo, $chatId);

            }

            break;

    }

}

?>

